% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compose.R
\name{compose}
\alias{compose}
\alias{\%>>>\%}
\title{Compose Functions}
\usage{
compose(...)

fst \%>>>\% snd
}
\arguments{
\item{...}{Functions or lists thereof to compose, in order of application.
Lists of functions are automatically spliced in.
\link[rlang:quasiquotation]{Unquoting} of names, via \code{!!} on the left-hand side
of \code{:=}, and \link[rlang:quasiquotation]{splicing}, via \code{!!!}, are supported.}

\item{fst, snd}{Functions. These may be optionally named using a colon (\code{:}),
e.g., \code{f \%>>>\% nm: g} names the \code{g}-component \code{"nm"} (see ‘Exceptions to
the Interpretation of Calls as Functions’).
\link[rlang:quasiquotation]{Quasiquotation} and the
\href{https://cran.r-project.org/package=magrittr}{\pkg{magrittr}} \code{`\%>\%`}
semantics are supported (see ‘Semantics of the Composition Operator’,
‘Quasiquotation’ and ‘Examples’).}
}
\value{
Function of class \code{CompositeFunction}, whose
\link[base:formals]{formals} are those of the first function applied (as a
closure).
}
\description{
To compose functions,
\itemize{
\item Use \code{compose()}:\preformatted{  compose(f, g, h, ...)
}

This makes the function that applies \code{f}, \emph{first}, then \code{g}, then \code{h}, etc.
It has the \link[base:formals]{formals} of the first function applied (namely
\code{f}). Thus\preformatted{  compose(paste, toupper)
}

is equivalent to the function\preformatted{  function(..., sep = " ", collapse = NULL) {
    toupper(paste(..., sep = sep, collapse = collapse))
  }
}
\item Use \code{`\%>>>\%`}:\preformatted{  f \%>>>\% g \%>>>\% h \%>>>\% ...
}

It comprehends both the semantics of the
\href{https://cran.r-project.org/package=magrittr}{\pkg{magrittr}} \code{`\%>\%`}
operator and \link[rlang:quasiquotation]{quasiquotation}. Thus, assuming \code{sep}
has the value \code{""},\preformatted{  sample \%>>>\% paste(collapse = !!sep)
}

is equivalent to the function\preformatted{  function(x, size, replace = FALSE, prob = NULL) {
    paste(sample(x, size, replace, prob), collapse = "")
  }
}
}

Use \code{\link[=as.list]{as.list()}} to recover the list of composite functions. For example, both\preformatted{  as.list(compose(paste, capitalize = toupper))

  as.list(paste \%>>>\% capitalize: toupper)
}

return the (named) list of functions \code{list(paste, capitalize = toupper)}.
}
\section{Semantics of the Composition Operator}{

The \code{`\%>>>\%`} operator adopts the semantics of the
\href{https://cran.r-project.org/package=magrittr}{\pkg{magrittr}} \code{`\%>\%`}
operator:
\enumerate{
\item \strong{Bare names are matched to functions} — For example, in a composition
like\preformatted{  ... \%>>>\% foo \%>>>\% ...
}

the ‘\code{foo}’ is matched to the function of that name.
\item \strong{Function calls are interpreted as a unary function of a point (\code{.})}
— A \emph{call} is interpreted as a \emph{function} (of a point) in one of two
ways:
\itemize{
\item If the point matches an argument value, the call is literally
interpreted as the body of the function. For example, in the
compositions\preformatted{  ... \%>>>\% foo(x, .) \%>>>\% ...

  ... \%>>>\% foo(x, y = .) \%>>>\% ...
}

the ‘\code{foo(x, .)}’, resp. ‘\code{foo(x, y = .)}’, is interpreted as the
function \code{function(..., . = ..1) foo(x, .)}, resp.
\code{function(..., . = ..1) foo(x, y = .)}.
\item Otherwise, the call is regarded as implicitly having the point as its
first argument before being interpreted as the body of the function.
For example, in the compositions\preformatted{  ... \%>>>\% foo(x) \%>>>\% ...

  ... \%>>>\% foo(x, y(.)) \%>>>\% ...
}

the ‘\code{foo(x)}’, resp. ‘\code{foo(x, y(.))}’, is interpreted as the function
\code{function(..., . = ..1) foo(., x)}, resp.
\code{function(..., . = ..1) foo(., x, y(.))}.
}
\item \strong{Expressions \code{{...}} are interpreted as a function of a point (\code{.})} —
For example, in a composition\preformatted{  ... \%>>>\% {
    foo(.)
    bar(.)
  } \%>>>\% ...
}

the ‘\code{{foo(.); bar(.)}}’ is interpreted as the function
\code{function(..., . = ..1) {foo(.); bar(.)}}.

Curly braces are useful when you need to circumvent \code{`\%>>>\%`}’s
usual interpretation of function calls. For example, in a composition\preformatted{  ... \%>>>\% {foo(x, y(.))} \%>>>\% ...
}

the ‘\code{{foo(x, y(.))}}’ is interpreted as the function
\code{function(..., . = ..1) foo(x, y(.))}—there is no point as first
argument to \code{foo}.
}

\subsection{Exceptions to the Interpretation of Calls as Functions}{
As a matter of convenience, some exceptions are made to the above
interpretation of calls as functions:
\itemize{
\item \strong{Parenthesis} (\code{\link[base:Paren]{(}}) applies grouping. (In R, \code{`(`}
is indeed a function.) In particular, expressions within parentheses are
literally interpreted.
\item \strong{Colon} (\code{\link[base:Colon]{:}}) applies \emph{naming}, according to the syntax
‘\code{<name>: <function>}’, where ‘\code{<function>}’ is interpreted according
to the semantics of \code{`\%>>>\%`}. For example, in\preformatted{  ... \%>>>\% aName: foo \%>>>\% ...
}

the function \code{foo} is named \code{"aName"}.
\item \strong{\code{\link[=fn]{fn()}}}, \strong{\link[base:ns-dblcolon]{namespace operators}} (\code{`::`},
\code{`:::`}) and \strong{\link[base:Extract]{extractors}} (\code{`$`}, \code{`[[`},
\code{`[`}) are literally interpreted. This allows for list extractors to
be applied to composite functions appearing in a \code{`\%>>>\%`} call (see
‘Operate on Composite Functions as List-Like Objects’). For example, the
compositions\preformatted{  paste \%>>>\% tolower

  paste \%>>>\% base::tolower

  (paste \%>>>\% toupper)[[1]] \%>>>\% tolower
}

are equivalent functions.
}
}
}

\section{Quasiquotation}{

The \code{`\%>>>\%`} operator supports Tidyverse
\link[rlang:quasiquotation]{unquoting} (via \code{!!}). Use it to:
\itemize{
\item \strong{Enforce immutability} — For example, by unquoting \code{res} in\preformatted{  res <- "result"
  get_result <- identity \%>>>\% lapply(`[[`, !!res)
}

you ensure that the function \code{get_result()} always extracts the component
named \code{"result"}, even if the binding \code{res} changes its value or is
removed altogether.
\item \strong{Interpret the point (\code{.}) in the lexical scope} — Even though
\code{`\%>>>\%`} interprets ‘\code{.}’ as a function argument, you can still
reference an object of that name via unquoting. For example,\preformatted{  . <- "point"
  is_point <- identity \%>>>\% {. == !!.}
}

determines a function that checks for equality with the string \code{"point"}.
\item \strong{Name composite functions, programmatically} — For example, unquoting
\code{nm} in\preformatted{  nm <- "aName"
  ... \%>>>\% !!nm: foo \%>>>\% ...
}

names the ‘\code{foo}’-component of the resulting composite function
\code{"aName"}.
\item \strong{Accelerate functions by fixing constant dependencies} — For example,
presuming the value of the call \code{f()} is \emph{constant} and that \code{g} is a
\emph{pure} function (meaning that its return value depends only on its
input), both\preformatted{  ... \%>>>\% g(f()) \%>>>\% ...

  ... \%>>>\% g(!!f()) \%>>>\% ...
}

would be functions yielding the same values. But the first would compute
\code{f()} anew with each call, whereas the second would simply depend on a
fixed, pre-computed value of \code{f()}.
}
}

\section{Operate on Composite Functions as List-Like Objects}{

You can think of a composite function as embodying the (possibly nested)
structure of its list of constituent functions. In fact, you can apply
familiar index and assignment operations to a composite function, as if it
were this list, getting a function in return. This enables you to leverage
composite functions as \emph{structured computations}.

\subsection{Indexing}{
For instance, the ‘\code{sum}’ in the following composite function\preformatted{  f <- abs \%>>>\% out: (log \%>>>\% agg: sum)
}

can be \link[base:Extract]{extracted} in the usual ways:\preformatted{  f[[2]][[2]]
  f[[c(2, 2)]]

  f$out$agg
  f[["out"]][["agg"]]
  f[["out"]]$agg

  f$out[[2]]
  f[[list("out", 2)]]
}

The last form of indexing with a mixed list is handy when you need to
create an index programmatically.

Additionally, you can excise sub-composite functions with
\code{\link[base:Extract]{[}}, \code{\link[=head]{head()}}, \code{\link[=tail]{tail()}}. For example:
\itemize{
\item Both \code{f[1]} and \code{head(f, 1)} get the ‘\code{abs}’ as a composite function,
namely \code{compose(abs)}
\item \code{f[2:1]} reverses the order of the top-level functions to yield\preformatted{  out: (log \%>>>\% agg: sum) \%>>>\% abs
}
\item \code{f$out[c(FALSE, TRUE)]} gets the ‘\code{sum}’ as a (named) composite function
}
}

\subsection{Subset Assignment}{
Similarily, subset assignment works as it does for lists. For instance, you
can replace the ‘\code{sum}’ with the identity function:\preformatted{  f[[2]][[2]] <- identity

  f$out$agg <- identity
  f[["out"]][["agg"]] <- identity

  f$out[[2]] <- identity
  f[[list("out", 2)]] <- identity
}

Multiple constituent functions can be reassigned using
\code{\link[base:Extract]{[<-}}. For example\preformatted{  f[2] <- list(log)

  f["out"] <- list(log)

  f[c(FALSE, TRUE)] <- list(log)
}

all replace the second constituent function with \code{log}, so that \code{f} becomes
\code{abs \%>>>\% log}.
}

\subsection{Other List Methods}{
The generic methods \code{\link[=unlist]{unlist()}}, \code{\link[=length]{length()}}, \code{\link[=names]{names()}} also apply to
composite functions. In conjunction with \code{compose()}, you can use
\code{unlist()} to “flatten” compositions. For example\preformatted{  compose(unlist(f, use.names = FALSE))
}

gives a function that is identical to\preformatted{  abs \%>>>\% log \%>>>\% sum
}

}
}

\section{Composite Functions Balance Speed and Complexity}{

The speed of a composite function made by \code{compose()} or \code{`\%>>>\%`}—regardless of its nested depth—is on par with a manually constructed
\emph{serial} composition. This is because \code{compose()} and \code{`\%>>>\%`} are
\strong{associative}, semantically and operationally. For instance, triple
compositions,\preformatted{  compose(f, g, h)
  f \%>>>\% g \%>>>\% h

  compose(f, compose(g, h))
  f \%>>>\% (g \%>>>\% h)

  compose(compose(f, g), h)
  (f \%>>>\% g) \%>>>\% h
}

are all implemented as the \emph{same function}—lists of functions are
automatically “flattened” when composed.

Nevertheless, the original nested structure of constituent functions is
faithfully recovered by \code{\link[=as.list]{as.list()}}. In particular, \code{as.list()} and
\code{compose()} are \strong{mutually invertible}: \code{as.list(compose(fs))} is the same
as \code{fs}, when \code{fs} is a (nested) list of functions. (But note that the
names of the list of composite functions is always a character vector; it
is never \code{NULL}.)
}

\examples{
# Functions are applied in the order in which they are listed
inv <- partial(`/`, 1)  # reciprocal
f0 <- compose(abs, log, inv)
stopifnot(all.equal(f0(-2), 1 / log(abs(-2))))

# Alternatively, compose using the `\%>>>\%` operator
f1 <- abs \%>>>\% log \%>>>\% {1 / .}
stopifnot(all.equal(f1(-2), f0(-2)))

\dontrun{
# Transform a function to a JSON function
library(jsonlite)

# By composing higher-order functions:
jsonify <- {fromJSON \%>>>\% .} \%>>>\% {. \%>>>\% toJSON}

# By directly composing with input/output transformers:
jsonify <- fn(f ~ fromJSON \%>>>\% f \%>>>\% toJSON)}

# Formals of initial function are preserved
add <- function(a, b = 0) a + b
stopifnot(identical(formals(compose(add, inv)), formals(add)))

# Compositions can be provided by lists, in several equivalent ways
f2 <- compose(list(abs, log, inv))
f3 <- compose(!!! list(abs, log, inv))
f4 <- compose(abs, list(log, inv))
f5 <- compose(abs, !!! list(log, inv))
stopifnot(
  all.equal(f2, f0), all.equal(f2(-2), f0(-2)),
  all.equal(f3, f0), all.equal(f3(-2), f0(-2)),
  all.equal(f4, f0), all.equal(f4(-2), f0(-2)),
  all.equal(f5, f0), all.equal(f5(-2), f0(-2))
)

# compose() and as.list() are mutally invertible
f6 <- compose(abs, as.list(compose(log, inv)))
stopifnot(
  all.equal(f6, f0), all.equal(f6(-2), f0(-2))
)
fs <- list(abs, log, inv)
stopifnot(all.equal(check.attributes = FALSE,
  as.list(compose(fs)), fs,
))

# `\%>>>\%` supports names, magrittr `\%>\%` semantics, and quasiquotation
sep <- ""
scramble <- shuffle: sample \%>>>\% paste(collapse = !!sep)
nonsense <- scramble(letters)
stopifnot(
  nchar(nonsense) == 26L,
  identical(letters, sort(strsplit(nonsense, sep)[[1]])),
  identical(scramble$shuffle, sample)
)

}
\seealso{
\code{\link[=constant]{constant()}}; combined with \code{`\%>>>\%`}, this provides a lazy,
structured alternative to the
\href{https://cran.r-project.org/package=magrittr}{\pkg{magrittr}} \code{`\%>\%`}
operator.
}
