#' Import MCMC samples into a ggs object than can be used by all ggs_* graphical functions.
#'
#' This function manages MCMC samples from different sources (JAGS, MCMCpack, STAN -both via rstan and via csv files-) and converts them into a data frame. The resulting data frame has four columns (Iteration, Parameter, value, Chain) and seven attributes (nChains, nParameters, nIterations, nBurnin, nThin, description and parallel). The ggs object returned is then used as the input of the ggs_* functions to actually plot the different convergence diagnostics.
#'
#' @param S Either a \code{mcmc.list} object with samples from JAGS, a \code{mcmc} object with samples from MCMCpack, a \code{stanfit} object with samples from rstan, or a list with the filenames of \code{csv} files generated by stan outside rstan (where the order of the files is assumed to be the order of the chains). ggmcmc guesses what is the original object and tries to import it accordingly.
#' @param description Character vector giving a short descriptive text that identifies the model.
#' @param burnin Logical or numerical value. When logical and TRUE, the number of samples in the burnin period will be taken into account, if it can be guessed by the extracting process. Otherwise, iterations will start counting from 1. Defaults to FALSE. If a numerical vector is given, the user then supplies the length of the burnin period. 
#' @param inc_warmup Logical. When dealing with stanfit objects from rstan, logical value whether the warmup samples are included. Defaults to FALSE.
#' @param stan_include_auxiliar Logical value to include "lp__" parameter in rstan, and "lp__", "treedepth__" and "stepsize__" in stan running without rstan. Defaults to FALSE.
#' @param parallel Logical value for using parallel computing when managing the data frame in other functions. Defaults to TRUE, although it has not been fully tested yet.
#' @export
#' @return D A data frame with the data arranged and ready to be used by the rest of the \code{ggmcmc} functions. The data frame has four columns, namely: Iteration, Parameter, value and Chain, and seven attributes: nChains, nParameters, nIterations, nBurnin, nThin, description and parallel.
#' @examples
#' # Assign 'D' to be a data frame suitable for \code{ggmcmc} functions from 
#' # a coda object called S
#' data(samples)
#' D <- ggs(S)        # S is a coda object
ggs <- function(S, description=NA, burnin=FALSE, inc_warmup=FALSE, stan_include_auxiliar=FALSE, parallel=TRUE) {
  #
  # Manage stanfit obcjets
  # Manage stan output first because it is firstly converted into an mcmc.list
  #
  if (class(S)=="stanfit") { 
    if (inc_warmup) warning("inc_warmup must be 'FALSE', so it is ignored.")
    S <- as.array(S)
    # If someday rstan gets into CRAN the more elegant 
    # rstan::extract(S, inc_warmup=inc_warmup) may be used
    S <- do.call(mcmc.list, alply(S, 2, coda::mcmc))
    # Exclude, by default, lp parameter
    if (!stan_include_auxiliar) {
      S <- S[,1:(dim(S[[1]])[2]-1)] # delete the last column, the last parameter, which is lp__
    }
  }
  #
  # Manage csv files than contain stan samples
  # Also converted first to an mcmc.list
  #
  if (class(S)=="list") {
    S.out <- coda::mcmc.list()
    for (i in 1:length(S)) {
      S.out[[i]] <- coda::mcmc(
        read.table(S[[i]], sep=",", header=TRUE, colClasses="numeric"))
    }
    S <- S.out
    # Exclude, by default, lp parameter
    if (!stan_include_auxiliar) {
      S <- S[,-c(1:3)] # delete the last column, the last parameter, which is lp__
    }
  }
  #
  # Manage mcmc.list and mcmc objects
  #
  if (class(S)=="mcmc.list" | class(S)=="mcmc") {  # JAGS typical output or MCMCpack
    lS <- length(S)
    D <- NULL
    if (lS == 1 | class(S)=="mcmc") { # Single chain or MCMCpack
      if (lS == 1 & class(S)=="mcmc.list") { # single chain
        s <- S[[1]]
      } else { # MCMCpack
        s <- S
      }
      # Process a single chain
      D <- cbind(ggs_chain(s), Chain=1)
      # Get information from mcpar (burnin period, thinning)
      nBurnin <- (attributes(s)$mcpar[1])-(1*attributes(s)$mcpar[3])
      nThin <- attributes(s)$mcpar[3]
    } else {
      # Process multiple chains
      for (l in 1:lS) {
        s <- S[l][[1]]
        D <- rbind(D, cbind(ggs_chain(s), Chain=l))
      }
      # Get information from mcpar (burnin period, thinning). Taking the last
      # chain is fine. All chains are assumed to have the same structure.
      nBurnin <- (attributes(s)$mcpar[1])-(1*attributes(s)$mcpar[3])
      nThin <- attributes(s)$mcpar[3]
    }
    # Set several attributes to the object, to avoid computations afterwards
    # Number of chains
    attr(D, "nChains") <- lS
    # Number of parameters
    attr(D, "nParameters") <- length(unique(D$Parameter))
    # Number of Iterations really present in the sample
    attr(D, "nIterations") <- max(D$Iteration)
    # Number of burning periods previously
    attr(D, "nBurnin") <- nBurnin
    # Thinning interval
    attr(D, "nThin") <- nThin
    # Descriptive text
    if (is.character(description)) { # if the description is given, us it when it is a character string
      attr(D, "description") <- description
    } else {
      if (!is.na(description)) { # if it is not a character string and not NA, show an informative message
        print("description is not a text string. The name of the imported object is used instead.")
      }
      attr(D, "description") <- as.character(sys.call()[2]) # use the name of the source object
    }
    # Whether parallel computing is desired
    attr(D, "parallel") <- parallel
    return(D)
  } else {
    stop("ggs is not able to transform the input object into a ggs object suitable for ggmcmc.")
  }
}

#' Auxiliary function that extracts information from a single chain.
#'
#' @param s a single chain to convert into a data frame
#' @return D data frame with the chain arranged
ggs_chain <- function(s) {
  # Get the names of the chains, the number of samples and the vector of
  # iterations
  name.chains <- dimnames(s)[[2]]
  n.samples <- dim(s)[1]
  iter <- 1:n.samples

  # Prepare the dataframe
  d <- data.frame(Iteration=iter, as.matrix(unclass(s)))
  D <- melt(d, id.vars=c("Iteration"), variable_name="Parameter")

  # Revert the name of the parameters to their original names
  levels(D$Parameter) <- name.chains

  # Return the modified data frame
  return(D)
}
