% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/project_matrix.R
\name{project_matrix}
\alias{project_matrix}
\title{Project matrix after optimization}
\usage{
project_matrix(S, perm, precomputed_equal_indices = NULL)
}
\arguments{
\item{S}{A square matrix to be projected.
The covariance estimator.
(See the same parameter in \code{\link[=gips]{gips()}} function).}

\item{perm}{A permutation. Generator of a permutation group.
Either of a \code{gips_perm} or a \code{permutations::cycle} class.}

\item{precomputed_equal_indices}{This parameter is for internal use only.}
}
\value{
Returns the matrix \code{S} projected on the space of symmetrical matrices invariant
by a cyclic group generated by \code{perm}. See Details for more.
}
\description{
After the MAP permutation was found with \code{\link[=find_MAP]{find_MAP()}},
use this permutation to approximate the covariance matrix
with larger statistical confidence.
}
\details{
Project matrix on the space of symmetrical matrices invariant
by a cyclic group generated by \code{perm}.
This implements the formal
\href{https://arxiv.org/abs/2004.03503}{Definition 3 from references}.

When \code{S} is the sample covariance matrix (output of \code{cov()} function, see
examples), then \code{S} is the \strong{unbiased estimator} of the covariance matrix.
However, the \strong{maximum likelihood estimator} of the covariance matrix is
\code{S*(n-1)/(n)}, unless n < p, when the
\strong{maximum likelihood estimator does not exist}. For more information, see
\href{https://en.wikipedia.org/wiki/Estimation_of_covariance_matrices}{Wikipedia - Estimation of covariance matrices}.

The maximum likelihood estimator differs when one knows the covariance
matrix is \strong{invariant under some permutation}. This estimator will
not only be symmetric but also have some values repeated (see examples and
\href{https://arxiv.org/abs/2004.03503}{Corollary 12 from references}).

The estimator will be invariant under the given permutation. Also, it
will \strong{need fewer observations} for the maximum likelihood estimator to
exist (see \strong{Project Matrix - Equation (6)} section in
\code{vignette("Theory", package = "gips")} or in its
\href{https://przechoj.github.io/gips/articles/Theory.html}{pkgdown page}).
For some permutations, even \eqn{n = 2} could be enough.
The minimal number of observations needed are named \code{n0} and
can be calculated by \code{\link[=summary.gips]{summary.gips()}}.

For more details, see the \strong{Project Matrix - Equation (6)}
section in \code{vignette("Theory", package = "gips")} or in its
\href{https://przechoj.github.io/gips/articles/Theory.html}{pkgdown page}.
}
\examples{
p <- 6
gperm <- gips_perm(permutations::as.word(c(4, 3, 2, 1, 5)), p) # permutation (1,4)(2,3)(5)(6)

number_of_observations <- 10
X <- matrix(rnorm(p * number_of_observations), number_of_observations, p)
S <- cov(X)
projected_S <- project_matrix(S, perm = gperm)
projected_S
# The value in [1,1] is the same as in [4,4]; also, [2,2] and [3,3];
  # also [1,2] and [4,3]; also, [1,5] and [4,5]; and so on

# Plot the projected matrix:
g <- gips(S, number_of_observations, perm = gperm)
plot(g, type = "heatmap")

# Find the MAP Estimator
g_MAP <- find_MAP(g, max_iter = 10, show_progress_bar = FALSE, optimizer = "Metropolis_Hastings")
S_MAP <- project_matrix(S, perm = g_MAP[[1]])
S_MAP
plot(g_MAP, type = "heatmap")
}
\seealso{
\itemize{
\item \href{https://en.wikipedia.org/wiki/Estimation_of_covariance_matrices}{Wikipedia - Estimation of covariance matrices}
\item \strong{Project Matrix - Equation (6)} section of
\code{vignette("Theory", package = "gips")} or its
\href{https://przechoj.github.io/gips/articles/Theory.html}{pkgdown page} -
A place to learn more about the math
behind the \code{gips} package and see
more examples of \code{project_matrix()}.
\item \code{\link[=find_MAP]{find_MAP()}} - The function that finds
the Maximum A Posteriori (MAP) Estimator
for a given \code{gips} object.
After the MAP Estimator is found, the matrix
\code{S} can be projected on this permutation,
creating the MAP Estimator of the covariance matrix
(see examples).
\item \code{\link[=gips_perm]{gips_perm()}} - Constructor for the \code{perm} parameter.
\item \code{\link[=plot.gips]{plot.gips()}} - For \code{plot(g, type = 'heatmap')},
the \code{project_matrix()} is called (see examples).
\item \code{\link[=summary.gips]{summary.gips()}} - Can calculate the \code{n0}, the minimal
number of observations, so that the projected matrix
will be the MAP estimator of the covariance matrix.
}
}
