% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{nrgkw}
\alias{nrgkw}
\title{Enhanced Newton-Raphson Optimization for GKw Family Distributions}
\usage{
nrgkw(
  start = NULL,
  data = as.numeric(c()),
  family = "gkw",
  tol = 1e-06,
  max_iter = 100L,
  verbose = FALSE,
  optimization_method = "trust-region",
  enforce_bounds = TRUE,
  min_param_val = 1e-05,
  max_param_val = 1e+05,
  adaptive_scaling = TRUE,
  use_stochastic_perturbation = TRUE,
  get_num_hess = FALSE,
  multi_start_attempts = 3L,
  eigenvalue_hessian_reg = TRUE,
  max_backtrack = 20L,
  initial_trust_radius = 1
)
}
\arguments{
\item{start}{A numeric vector containing initial values for the parameters.
If NULL, automatic initialization is used based on the dataset characteristics.
The length and order must correspond to the selected \code{family}
(e.g., \code{c(alpha, beta, gamma, delta, lambda)} for "gkw"; \code{c(alpha, beta)} for "kw";
\code{c(gamma, delta)} for "beta").}

\item{data}{A numeric vector containing the observed data. All values must
be strictly between 0 and 1.}

\item{family}{A character string specifying the distribution family. One of
\code{"gkw"}, \code{"bkw"}, \code{"kkw"}, \code{"ekw"}, \code{"mc"},
\code{"kw"}, or \code{"beta"}. Default: \code{"gkw"}.}

\item{tol}{Convergence tolerance. The algorithm stops when the Euclidean norm
of the gradient is below this value, or if relative changes in parameters
or the negative log-likelihood are below this threshold across consecutive
iterations. Default: \code{1e-6}.}

\item{max_iter}{Maximum number of iterations allowed. Default: \code{100}.}

\item{verbose}{Logical; if \code{TRUE}, prints detailed progress information
during optimization, including iteration number, negative log-likelihood,
gradient norm, and step adjustment details. Default: \code{FALSE}.}

\item{optimization_method}{Character string specifying the optimization method:
"trust-region" (default), "newton-raphson", or "hybrid" (starts with trust-region,
switches to newton-raphson near convergence).}

\item{enforce_bounds}{Logical; if \code{TRUE}, parameter values are constrained
to stay within \code{min_param_val}, \code{max_param_val} (and \eqn{\delta \ge 0})
during optimization. Default: \code{TRUE}.}

\item{min_param_val}{Minimum allowed value for parameters constrained to be
strictly positive (\eqn{\alpha, \beta, \gamma, \lambda}). Default: \code{1e-5}.}

\item{max_param_val}{Maximum allowed value for all parameters. Default: \code{1e5}.}

\item{adaptive_scaling}{Logical; if \code{TRUE}, parameters are automatically scaled
to improve numerical stability. Default: \code{TRUE}.}

\item{use_stochastic_perturbation}{Logical; if \code{TRUE}, applies random perturbations
when optimization stalls. Default: \code{TRUE}.}

\item{get_num_hess}{Logical; if \code{TRUE}, computes and returns a numerical
approximation of the Hessian at the solution. Default: \code{FALSE}.}

\item{multi_start_attempts}{Integer specifying the number of different starting points
to try if initial optimization fails to converge. Default: \code{3}.}

\item{eigenvalue_hessian_reg}{Logical; if \code{TRUE}, uses eigenvalue-based
regularization for the Hessian matrix. Default: \code{TRUE}.}

\item{max_backtrack}{Integer specifying the maximum number of backtracking steps
in line search. Default: \code{20}.}

\item{initial_trust_radius}{Initial radius for trust region method. Default: \code{1.0}.}
}
\value{
A list object of class \code{gkw_fit} containing the following components:
\item{parameters}{A named numeric vector with the estimated parameters.}
\item{loglik}{The maximized value of the log-likelihood function.}
\item{iterations}{Number of iterations performed.}
\item{converged}{Logical flag indicating whether the algorithm converged successfully.}
\item{param_history}{A matrix where rows represent iterations and columns represent parameter values.}
\item{loglik_history}{A vector of log-likelihood values at each iteration.}
\item{gradient}{The gradient vector at the final parameter estimates.}
\item{hessian}{The analytical Hessian matrix at the final parameter estimates.}
\item{std_errors}{A named numeric vector of approximate standard errors for the parameters.}
\item{aic}{Akaike Information Criterion.}
\item{bic}{Bayesian Information Criterion.}
\item{aicc}{AIC with small sample correction.}
\item{n}{The sample size.}
\item{status}{A character string indicating the termination status.}
\item{z_values}{A named numeric vector of Z-statistics for parameter significance testing.}
\item{p_values}{A named numeric vector of two-sided p-values corresponding to the Z-statistics.}
\item{param_names}{A character vector of the parameter names.}
\item{family}{The distribution family used.}
\item{optimization_method}{The optimization method used.}
\item{numeric_hessian}{The numerically approximated Hessian at the solution (if requested).}
\item{condition_number}{The condition number of the final Hessian, a measure of parameter identifiability.}
\item{scaling_factors}{The scaling factors used for parameters (if adaptive scaling was enabled).}
}
\description{
An industrial-strength implementation of maximum likelihood estimation (MLE)
for the parameters of any distribution in the Generalized Kumaraswamy (GKw) family.
This function incorporates multiple advanced numerical techniques including trust region
methods, eigenvalue-based regularization, adaptive scaling, and sophisticated line
search to ensure robust convergence even for challenging datasets or extreme parameter values.
}
\details{
This enhanced algorithm provides robust parameter estimation for the Generalized
Kumaraswamy distribution and its subfamilies. The function implements several
advanced numerical optimization techniques to maximize the likelihood function
reliably even in difficult cases.

\subsection{The GKw Family of Distributions}{
The Generalized Kumaraswamy (GKw) distribution, introduced by Carrasco, Ferrari,
and Cordeiro (2010), is a flexible five-parameter continuous distribution defined
on the standard unit interval (0,1). Its probability density function is given by:

\deqn{f(x; \alpha, \beta, \gamma, \delta, \lambda) = \frac{\lambda\alpha\beta x^{\alpha-1}}{B(\gamma, \delta+1)}(1-x^{\alpha})^{\beta-1}[1-(1-x^{\alpha})^{\beta}]^{\gamma\lambda-1}\{1-[1-(1-x^{\alpha})^{\beta}]^{\lambda}\}^{\delta}}

where \eqn{\alpha, \beta, \gamma, \lambda > 0} and \eqn{\delta \geq 0} are the model
parameters, and \eqn{B(\gamma, \delta+1)} is the beta function.

The GKw distribution encompasses several important special cases:
\itemize{
\item\bold{GKw} (5 parameters): \eqn{\alpha, \beta, \gamma, \delta, \lambda}
\item\bold{BKw} (4 parameters): \eqn{\alpha, \beta, \gamma, \delta} (with \eqn{\lambda = 1})
\item\bold{KKw} (4 parameters): \eqn{\alpha, \beta, \delta, \lambda} (with \eqn{\gamma = 1})
\item\bold{EKw} (3 parameters): \eqn{\alpha, \beta, \lambda} (with \eqn{\gamma = 1, \delta = 0})
\item\bold{Mc}  (3 parameters): \eqn{\gamma, \delta, \lambda} (with \eqn{\alpha = 1, \beta = 1})
\item\bold{Kw}  (2 parameters): \eqn{\alpha, \beta} (with \eqn{\gamma = 1, \delta = 0, \lambda = 1})
\item\bold{Beta}(2 parameters): \eqn{\gamma, \delta} (with \eqn{\alpha = 1, \beta = 1, \lambda = 1})
}
}

\subsection{Trust Region Method with Levenberg-Marquardt Algorithm}{
The trust region approach restricts parameter updates to a region where the quadratic
approximation of the objective function is trusted to be accurate. This algorithm
implements the Levenberg-Marquardt variant, which solves the subproblem:

\deqn{\min_p m_k(p) = -\nabla \ell(\theta_k)^T p + \frac{1}{2}p^T H_k p}
\deqn{\text{subject to } \|p\| \leq \Delta_k}

where \eqn{\nabla \ell(\theta_k)} is the gradient, \eqn{H_k} is the Hessian, and \eqn{\Delta_k}
is the trust region radius at iteration \eqn{k}.

The Levenberg-Marquardt approach adds a regularization parameter \eqn{\lambda} to the
Hessian, solving:

\deqn{(H_k + \lambda I)p = -\nabla \ell(\theta_k)}

The parameter \eqn{\lambda} controls the step size and direction:
\itemize{
\item When \eqn{\lambda} is large, the step approaches a scaled steepest descent direction.
\item When \eqn{\lambda} is small, the step approaches the Newton direction.
}

The algorithm dynamically adjusts \eqn{\lambda} based on the agreement between the quadratic model and the actual function:

\deqn{\rho_k = \frac{f(\theta_k) - f(\theta_k + p_k)}{m_k(0) - m_k(p_k)}}

The trust region radius is updated according to:
\itemize{
\item If \eqn{\rho_k < 0.25}, reduce the radius: \eqn{\Delta_{k+1} = 0.5\Delta_k}
\item If \eqn{\rho_k > 0.75} and \eqn{\|p_k\| \approx \Delta_k}, increase the radius: \eqn{\Delta_{k+1} = 2\Delta_k}
\item Otherwise, leave the radius unchanged: \eqn{\Delta_{k+1} = \Delta_k}
}

The step is accepted if \eqn{\rho_k > \eta} (typically \eqn{\eta = 0.1}).
}

\subsection{Eigenvalue-Based Hessian Regularization}{
For the Newton-Raphson method to converge, the Hessian matrix must be positive definite.
This algorithm uses eigendecomposition to create a positive definite approximation that
preserves the Hessian's structure:

\deqn{H = Q\Lambda Q^T}

where \eqn{Q} contains the eigenvectors and \eqn{\Lambda} is a diagonal matrix of eigenvalues.

The regularized Hessian is constructed by:

\deqn{\tilde{H} = Q\tilde{\Lambda}Q^T}

where \eqn{\tilde{\Lambda}} contains modified eigenvalues:

\deqn{\tilde{\lambda}_i = \max(\lambda_i, \epsilon)}

with \eqn{\epsilon} being a small positive threshold (typically \eqn{10^{-6}}).

This approach is superior to diagonal loading (\eqn{H + \lambda I}) as it:
\itemize{
\item Preserves the eigenvector structure of the original Hessian
\item Minimally modifies the eigenvalues needed to ensure positive definiteness
\item Better maintains the directional information in the Hessian
}
}

\subsection{Parameter Scaling for Numerical Stability}{
When parameters have widely different magnitudes, optimization can become numerically
unstable. The adaptive scaling system transforms the parameter space to improve conditioning:

\deqn{\theta_i^{scaled} = s_i \theta_i}

where scaling factors \eqn{s_i} are determined by:
\itemize{
\item For large parameters (\eqn{|\theta_i| > 100}): \eqn{s_i = 100/|\theta_i|}
\item For small parameters (\eqn{0 < |\theta_i| < 0.01}): \eqn{s_i = 0.01/|\theta_i|}
\item Otherwise: \eqn{s_i = 1}
}

The optimization is performed in the scaled space, with appropriate transformations
for the gradient and Hessian:

\deqn{\nabla \ell(\theta^{scaled})_i = \frac{1}{s_i}\nabla \ell(\theta)_i}
\deqn{H(\theta^{scaled})_{ij} = \frac{1}{s_i s_j}H(\theta)_{ij}}

The final results are back-transformed to the original parameter space before being returned.
}

\subsection{Line Search with Wolfe Conditions}{
The line search procedure ensures sufficient decrease in the objective function when
taking a step in the search direction. The implementation uses Wolfe conditions which
include both:
\enumerate{
\item Sufficient decrease (Armijo) condition:
\deqn{f(\theta_k + \alpha p_k) \leq f(\theta_k) + c_1 \alpha \nabla f(\theta_k)^T p_k}
\item Curvature condition:
\deqn{|\nabla f(\theta_k + \alpha p_k)^T p_k| \leq c_2 |\nabla f(\theta_k)^T p_k|}
}

where \eqn{0 < c_1 < c_2 < 1}, typically \eqn{c_1 = 10^{-4}} and \eqn{c_2 = 0.9}.

The step length \eqn{\alpha} is determined using polynomial interpolation:
\itemize{
\item First iteration: quadratic interpolation
\item Subsequent iterations: cubic interpolation using function and derivative values
}

The cubic polynomial model has the form:
\deqn{a\alpha^3 + b\alpha^2 + c\alpha + d}

The algorithm computes coefficients from values at two points, then finds the minimizer
of this polynomial subject to bounds to ensure convergence.
}

\subsection{Adaptive Numerical Differentiation}{
When analytical derivatives are unreliable, the algorithm uses numerical differentiation
with adaptive step sizes based on parameter magnitudes:

\deqn{h_i = \max(h_{min}, \min(h_{base}, h_{base} \cdot |\theta_i|))}

where \eqn{h_{min}} is a minimum step size (typically \eqn{10^{-8}}), \eqn{h_{base}}
is a base step size (typically \eqn{10^{-5}}), and \eqn{\theta_i} is the parameter value.

For computing diagonal Hessian elements, the central difference formula is used:

\deqn{\frac{\partial^2 f}{\partial \theta_i^2} \approx \frac{f(\theta + h_i e_i) - 2f(\theta) + f(\theta - h_i e_i)}{h_i^2}}

For mixed partial derivatives:

\deqn{\frac{\partial^2 f}{\partial \theta_i \partial \theta_j} \approx \frac{f(\theta + h_i e_i + h_j e_j) - f(\theta + h_i e_i - h_j e_j) - f(\theta - h_i e_i + h_j e_j) + f(\theta - h_i e_i - h_j e_j)}{4h_i h_j}}

The algorithm validates numerical differentiation by comparing with existing gradients
and adaptively adjusts step sizes when discrepancies are detected.
}

\subsection{Stochastic Perturbation}{
To escape flat regions or local minima, the algorithm implements controlled stochastic
perturbation when progress stalls (detected by monitoring gradient norm changes):

\deqn{\theta_i^{new} = \theta_i + \Delta\theta_i}

where the perturbation \eqn{\Delta\theta_i} combines:
\itemize{
\item A directed component opposite to the gradient: \eqn{-\text{sign}(\nabla \ell_i) \cdot 0.05 \cdot |\theta_i|}
\item A random noise component: \eqn{U(-0.05|\theta_i|, 0.05|\theta_i|)}
}

The perturbation is applied when:
\itemize{
\item The relative change in gradient norm is below a threshold for several consecutive iterations
\item The algorithm appears to be stuck in a non-optimal region
}

The perturbation is accepted only if it improves the objective function value.
}

\subsection{Multi-Start Strategy}{
For particularly challenging optimization landscapes, the algorithm can employ multiple
starting points:

\itemize{
\item Initial values are generated using moment-based estimation and domain knowledge about each distribution family
\item Each initial point is randomly perturbed to explore different regions of the parameter space
\item Independent optimization runs are performed from each starting point
\item The best result (based on likelihood value and convergence status) is returned
}

This approach increases the probability of finding the global optimum or a high-quality
local optimum, particularly for complex models with many parameters.
}

\subsection{Advanced Parameter Initialization}{
Intelligent starting values are critical for convergence in complex models. The algorithm
uses data-driven initialization based on:

\itemize{
\item Method of moments estimators for beta parameters:
\deqn{\alpha + \beta = \frac{\bar{x}(1-\bar{x})}{s^2} - 1}
\deqn{\alpha = (\alpha + \beta)\bar{x}}

\item Transformations to Kumaraswamy parameters:
\deqn{a_{Kw} = \sqrt{\alpha_{Beta}}}
\deqn{b_{Kw} = \sqrt{\beta_{Beta}}}

\item Adjustments based on data skewness (detected via mean relative to 0.5)

\item Corrections based on data dispersion (range relative to (0,1) interval)
}

The transformations between beta and Kumaraswamy parameters leverage the similarities
between these distributions while accounting for their parametric differences.
}

\subsection{Hybrid Optimization Strategy}{
The algorithm can dynamically switch between trust region and Newton-Raphson methods
based on optimization progress:

\itemize{
\item Early iterations: trust region method for stability and global convergence properties
\item Later iterations (when close to optimum): Newton-Raphson with line search for quadratic convergence rate
}

The switching criteria are based on iteration count and gradient norm, with additional
logic to handle cases where one method consistently fails.
}
}
\section{Warning}{

Although this implementation is highly robust, fitting complex distributions can still be challenging.
For best results:
\itemize{
\item{Try multiple starting values if results seem suboptimal}
\item{Examine diagnostic information carefully, especially condition numbers and standard errors}
\item{Be cautious of parameter estimates at or very near boundaries}
\item{Consider model simplification if convergence is consistently problematic}
\item{For the full GKw model with 5 parameters, convergence may be sensitive to starting values}
\item{High condition numbers (>1e6) may indicate parameter redundancy or weak identifiability}
}
}

\examples{
\donttest{
# Generate sample data from a Beta(2,5) distribution for testing
set.seed(123)
sample_data <- rbeta_(200, 2, 5)

# Automatic initialization (recommended for beginners)
result_auto <- nrgkw(NULL, sample_data, family = "beta", verbose = FALSE)
print(result_auto$parameters)
print(result_auto$loglik)

# Compare different optimization methods
methods <- c("trust-region", "newton-raphson", "hybrid")
results <- list()

for (method in methods) {
  results[[method]] <- nrgkw(NULL, sample_data, family = "beta",
                               optimization_method = method)
  cat(sprintf("Method: \%s, AIC: \%.4f\n", method, results[[method]]$aic))
}

# Fit the full GKw model with diagnostic information
gkw_result <- nrgkw(NULL, sample_data, family = "gkw",
                      verbose = FALSE, get_num_hess = TRUE)

# Examine parameter identifiability through condition number
cat(sprintf("Condition number: \%.2e\n", gkw_result$condition_number))
print(gkw_result)

# Compare with simpler models using information criteria
cat("Information criteria comparison:\n")
cat(sprintf("GKw: AIC=\%.4f, BIC=\%.4f\n", gkw_result$aic, gkw_result$bic))
cat(sprintf("Beta: AIC=\%.4f, BIC=\%.4f\n",
           results[["trust-region"]]$aic, results[["trust-region"]]$bic))
}
}
\references{
Carrasco, J. M. F., Ferrari, S. L. P., & Cordeiro, G. M. (2010). A new generalized Kumaraswamy
distribution. arXiv preprint arXiv:1004.0911.

Nocedal, J., & Wright, S. J. (2006). Numerical Optimization (2nd ed.). Springer.

Conn, A. R., Gould, N. I. M., & Toint, P. L. (2000). Trust Region Methods. MPS-SIAM Series on Optimization.

Kumaraswamy, P. (1980). A generalized probability density function for double-bounded
random processes. Journal of Hydrology, 46(1-2), 79-88.
}
\author{
Enhanced by Lopes, J. E.
}
\keyword{beta}
\keyword{distribution}
\keyword{kumaraswamy}
\keyword{likelihood}
\keyword{mcdonald}
\keyword{mle}
\keyword{newton-raphson}
\keyword{optimization}
