% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coxnet.deviance.R
\name{coxnet.deviance}
\alias{coxnet.deviance}
\title{Compute deviance for Cox model}
\usage{
coxnet.deviance(
  pred = NULL,
  y,
  x = NULL,
  offset = NULL,
  weights = NULL,
  std.weights = TRUE,
  beta = NULL
)
}
\arguments{
\item{pred}{Fit vector or matrix (usually from glmnet at a particular
lambda or a sequence of lambdas).}

\item{y}{Survival response variable, must be a \code{Surv} or
\code{stratifySurv} object.}

\item{x}{Optional \code{x} matrix, to be supplied if \code{pred = NULL}.}

\item{offset}{Optional offset vector.}

\item{weights}{Observation weights (default is all equal to 1).}

\item{std.weights}{If TRUE (default), observation weights are standardized
to sum to 1.}

\item{beta}{Optional coefficient vector/matrix, to be supplied if
\code{pred = NULL}.}
}
\value{
A vector of deviances, one for each column of predictions.
}
\description{
Compute the deviance (-2 log partial likelihood) for Cox model.
}
\details{
Computes the deviance for a single set of predictions, or for a matrix
of predictions. The user can either supply the predictions
directly through the \code{pred} option, or by supplying the \code{x} matrix
and \code{beta} coefficients. Uses the Breslow approach to ties.

The function first checks if \code{pred} is passed: if so, it is used as
the predictions. If \code{pred} is not passed but \code{x} and \code{beta}
are passed, then these values are used to compute the predictions. If
neither \code{x} nor \code{beta} are passed, then the predictions are all
taken to be 0.

\code{coxnet.deviance()} is a wrapper: it calls the appropriate internal
routine based on whether the response is right-censored data or
(start, stop] survival data.
}
\examples{
set.seed(1)
eta <- rnorm(10)
time <- runif(10, min = 1, max = 10)
d <- ifelse(rnorm(10) > 0, 1, 0)
y <- survival::Surv(time, d)
coxnet.deviance(pred = eta, y = y)

# if pred not provided, it is set to zero vector
coxnet.deviance(y = y)

# example with x and beta
x <- matrix(rnorm(10 * 3), nrow = 10)
beta <- matrix(1:3, ncol = 1)
coxnet.deviance(y = y, x = x, beta = beta)

# example with (start, stop] data
y2 <- survival::Surv(time, time + runif(10), d)
coxnet.deviance(pred = eta, y = y2)

# example with strata
y2 <- stratifySurv(y, rep(1:2, length.out = 10))
coxnet.deviance(pred = eta, y = y2)

}
\seealso{
\code{coxgrad}
}
\keyword{Cox}
\keyword{model}
