% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/confint-methods.R
\name{confint.fderiv}
\alias{confint.fderiv}
\title{Point-wise and simultaneous confidence intervals for derivatives of smooths}
\usage{
\method{confint}{fderiv}(
  object,
  parm,
  level = 0.95,
  type = c("confidence", "simultaneous"),
  nsim = 10000,
  ncores = 1L,
  ...
)
}
\arguments{
\item{object}{an object of class \code{"fderiv"} containing the estimated
derivatives.}

\item{parm}{which parameters (smooth terms) are to be given intervals as a
vector of terms. If missing, all parameters are considered.}

\item{level}{numeric, \verb{0 < level < 1}; the confidence level of the
point-wise or simultaneous interval. The default is \code{0.95} for a 95\%
interval.}

\item{type}{character; the type of interval to compute. One of \code{"confidence"}
for point-wise intervals, or \code{"simultaneous"} for simultaneous intervals.}

\item{nsim}{integer; the number of simulations used in computing the
simultaneous intervals.}

\item{ncores}{number of cores for generating random variables from a
multivariate normal distribution. Passed to \code{\link[mvnfast:rmvn]{mvnfast::rmvn()}}.
Parallelization will take place only if OpenMP is supported (but appears
to work on Windows with current \code{R}).}

\item{...}{additional arguments for methods}
}
\value{
a data frame with components:
\enumerate{
\item \code{term}; factor indicating to which term each row relates,
\item \code{lower}; lower limit of the confidence or simultaneous interval,
\item \code{est}; estimated derivative
\item \code{upper}; upper limit of the confidence or simultaneous interval.
}
}
\description{
Calculates point-wise confidence or simultaneous intervals for the first
derivatives of smooth terms in a fitted GAM.
}
\examples{
load_mgcv()
\dontshow{
op <- options(pillar.sigfig = 2, cli.unicode = FALSE)
}
dat <- data_sim("eg1", n = 1000, dist = "normal", scale = 2, seed = 2)
mod <- gam(y ~ s(x0) + s(x1) + s(x2) + s(x3), data = dat, method = "REML")

# new data to evaluate the derivatives at, say over the middle 50\% of range
# of each covariate
middle <- function(x, n = 25, coverage = 0.5) {
  v <- (1 - coverage) / 2
  q <- quantile(x, prob = c(0 + v, 1 - v), type = 8)
  seq(q[1], q[2], length = n)
}
new_data <- sapply(dat[c("x0", "x1", "x2", "x3")], middle)
new_data <- data.frame(new_data)
## first derivatives of all smooths...
fd <- fderiv(mod, newdata = new_data)

## point-wise interval
ci <- confint(fd, type = "confidence")
ci

## simultaneous interval for smooth term of x2
\dontshow{
set.seed(42)
}
x2_sint <- confint(fd, parm = "x2", type = "simultaneous",
                   nsim = 10000, ncores = 2)
\donttest{
x2_sint
}
\dontshow{options(op)}
}
\author{
Gavin L. Simpson
}
