% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ro.R
\name{ro}
\alias{ro}
\title{Rolling Origin}
\usage{
ro(data, h = 10, origins = 10, call, value = NULL, ci = FALSE,
  co = FALSE, silent = TRUE, parallel = FALSE)
}
\arguments{
\item{data}{Data vector or ts object passed to the function.}

\item{h}{The forecasting horizon.}

\item{origins}{The number of rolling origins.}

\item{call}{The call that is passed to the function. The call must be in quotes.
Example: \code{"forecast(ets(data),h)"}. Here \code{data} shows where the data is
and \code{h} defines where the horizon should be passed in the \code{call}. Some
hidden parameters can also be specified in the call. For example, parameters
\code{counti}, \code{counto} and \code{countf} are used in the inner loop
and can be used for the regulation of exogenous variables sizes. See examples
for the details.}

\item{value}{The variable or set of variables returned by the \code{call}.
For example, "\code{mean}" for functions of \code{forecast} package. This can
also be a vector of variables. See examples for the details. If the parameter
is \code{NULL}, then all the values from the call are returned (could be really
messy!).}

\item{ci}{The parameter defines if the in-sample window size should be constant.
If \code{TRUE}, then with each origin one observation is added at the end of
series and another one is removed from the beginning.}

\item{co}{The parameter defines whether the holdout sample window size should
be constant. If \code{TRUE}, the rolling origin will stop when less than
\code{h} observations are left in the holdout.}

\item{silent}{If \code{TRUE}, nothing is printed out in the console.}

\item{parallel}{If \code{TRUE}, then the rolling origin is done in parallel.
WARNING! Packages \code{foreach} and either \code{doMC} (Linux only),
\code{doParallel} or \code{doSNOW} are needed in order to run the function
in parallel.}
}
\value{
Function returns the following variables:
\itemize{
\item{\code{actuals} - the matrix of actual values corresponding to the
produced forecasts from each origin.}
\item{\code{value} - the matrices with the produced data from each origin.
Name of each matrix corresponds to the names in the parameter \code{value}.}
}
}
\description{
The function does rolling origin for any forecasting function
}
\details{
This function produces rolling origin forecasts using the \code{data} and a
\code{call} passed as parameters. The function can do all of that either in
serial or in parallel, but it needs \code{foreach} and either \code{doMC}
(Linux only), \code{doParallel} or \code{doSNOW} packages installed in order
to do the latter.
}
\examples{

x <- rnorm(100,0,1)
ourCall <- "predict(arima(x=data,order=c(0,1,1)),n.ahead=h)"

# The default call and values
ourValue <- "pred"
ro(x, h=5, origins=5, ourCall, ourValue)

# You can also use dolar sign
ourValue <- "$pred"
# And you can have constant in-sample size
ro(x, h=5, origins=5, ourCall, ourValue, ci=TRUE)

# You can ask for several values
ourValue <- c("pred","se")
# And you can have constant holdout size
ro(x, h=5, origins=20, ourCall, ourValue, ci=TRUE, co=TRUE)

#### The following code will give exactly the same result as above,
#### but computed in parallel using all but 1 core of CPU:
\dontrun{ro(x, h=5, origins=20, ourCall, ourValue, ci=TRUE, co=TRUE, parallel=TRUE)}


#### A more complicated example using the for loop and
#### several time series
x <- matrix(rnorm(120*3,0,1), 120, 3)

## Form an array for the forecasts we will produce
## We will have 4 origins with 6-steps ahead forecasts
ourForecasts <- array(NA,c(6,4,3))

## Define models that need to be used for each series
ourModels <- list(c(0,1,1), c(0,0,1), c(0,1,0))

## This call uses specific models for each time series
ourCall <- "predict(arima(data, order=ourModels[[i]]), n.ahead=h)"
ourValue <- "pred"

## Start the loop. The important thing here is to use the same variable 'i' as in ourCall.
for(i in 1:3){
    ourdata <- x[,i]
    ourForecasts[,,i] <- ro(data=ourdata,h=6,origins=4,call=ourCall,
                            value=ourValue,co=TRUE,silent=TRUE)$pred
}

## ourForecasts array now contains rolling origin forecasts from specific
## models.

##### An example with exogenous variables
x <- rnorm(100,0,1)
xreg <- rnorm(100,0,1)

## 'counti' is used to define in-sample size of xreg,
## 'counto' - the size of the holdout sample of xreg

ourCall <- "predict(arima(x=data, order=c(0,1,1), xreg=xreg[counti]),
            n.ahead=h, newxreg=xreg[counto])"
ourValue <- "pred"
ro(x,h=5,origins=5,ourCall,ourValue)


## 'countf' is used to take xreg of the size corresponding to the whole
## sample on each iteration
## This is useful when working with functions from smooth package.
## The following call will return the forecasts from es() function of smooth.
\dontrun{ourCall <- "es(data=data, h=h, xreg=xreg[countf])"
ourValue <- "forecast"
ro(x,h=5,origins=5,ourCall,ourValue)}


}
\references{
\itemize{
\item Tashman, (2000) Out-of-sample tests of forecasting accuracy:
an analysis and review International Journal of Forecasting, 16,
pp. 437-450. \url{https://doi.org/10.1016/S0169-2070(00)00065-0}.
}
}
\author{
Yves Sagaert

Ivan Svetunkov, \email{ivan@svetunkov.ru}
}
\keyword{ts}
