% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gs_design_npe.R
\name{gs_design_npe}
\alias{gs_design_npe}
\title{Group sequential design computation with non-constant effect and information}
\usage{
gs_design_npe(
  theta = 0.1,
  theta0 = NULL,
  theta1 = NULL,
  info = 1,
  info0 = NULL,
  info1 = NULL,
  info_scale = c(0, 1, 2),
  alpha = 0.025,
  beta = 0.1,
  upper = gs_b,
  upar = qnorm(0.975),
  lower = gs_b,
  lpar = -Inf,
  test_upper = TRUE,
  test_lower = TRUE,
  binding = FALSE,
  r = 18,
  tol = 1e-06
)
}
\arguments{
\item{theta}{Natural parameter for group sequential design
representing expected incremental drift at all analyses;
used for power calculation.}

\item{theta0}{Natural parameter used for upper bound spending;
if \code{NULL}, this will be set to 0.}

\item{theta1}{Natural parameter used for lower bound spending;
if \code{NULL}, this will be set to \code{theta}
which yields the usual beta-spending.
If set to 0, spending is 2-sided under null hypothesis.}

\item{info}{Proportionate statistical information at
all analyses for input \code{theta}.}

\item{info0}{Proportionate statistical information
under null hypothesis, if different than alternative;
impacts null hypothesis bound calculation.}

\item{info1}{Proportionate statistical information
under alternate hypothesis;
impacts null hypothesis bound calculation.}

\item{info_scale}{The information scale for calculation.}

\item{alpha}{One-sided Type I error.}

\item{beta}{Type II error.}

\item{upper}{Function to compute upper bound.}

\item{upar}{Parameters passed to the function provided in \code{upper}.}

\item{lower}{Function to compare lower bound.}

\item{lpar}{Parameters passed to the function provided in \code{lower}.}

\item{test_upper}{Indicator of which analyses should include
an upper (efficacy) bound; single value of \code{TRUE} (default) indicates
all analyses; otherwise, a logical vector of the same length as \code{info}
should indicate which analyses will have an efficacy bound.}

\item{test_lower}{Indicator of which analyses should include an lower bound;
single value of \code{TRUE} (default) indicates all analyses;
single value \code{FALSE} indicates no lower bound; otherwise,
a logical vector of the same length as \code{info} should indicate which
analyses will have a lower bound.}

\item{binding}{Indicator of whether futility bound is binding;
default of \code{FALSE} is recommended.}

\item{r}{Integer value controlling grid for numerical integration
as in Jennison and Turnbull (2000); default is 18, range is 1 to 80.
Larger values provide larger number of grid points and greater accuracy.
Normally \code{r} will not be changed by the user.}

\item{tol}{Tolerance parameter for boundary convergence (on Z-scale).}
}
\value{
A tibble with columns analysis, bound, z, probability, theta, info, info0.
}
\description{
Derives group sequential design size,
bounds and boundary crossing probabilities based on proportionate
information and effect size at analyses.
It allows a non-constant treatment effect over time,
but also can be applied for the usual homogeneous effect size designs.
It requires treatment effect and proportionate statistical information
at each analysis as well as a method of deriving bounds, such as spending.
The routine enables two things not available in the gsDesign package:
\enumerate{
\item non-constant effect, 2) more flexibility in boundary selection.
For many applications, the non-proportional-hazards design function
\code{gs_design_nph()} will be used; it calls this function.
Initial bound types supported are 1) spending bounds,
\item fixed bounds, and 3) Haybittle-Peto-like bounds.
The requirement is to have a boundary update method that
can each bound without knowledge of future bounds.
As an example, bounds based on conditional power that
require knowledge of all future bounds are not supported by this routine;
a more limited conditional power method will be demonstrated.
Boundary family designs Wang-Tsiatis designs including
the original (non-spending-function-based) O'Brien-Fleming and Pocock designs
are not supported by \code{\link[=gs_power_npe]{gs_power_npe()}}.
}
}
\details{
The inputs \code{info} and \code{info0} should be
vectors of the same length with increasing positive numbers.
The design returned will change these by some constant scale
factor to ensure the design has power \code{1 - beta}.
The bound specifications in \code{upper}, \code{lower}, \code{upar}, \code{lpar}
will be used to ensure Type I error and other boundary properties are as specified.
}
\section{Specification}{

\if{latex}{
 \itemize{
   \item Validate if input info is a numeric vector  or NULL, if non-NULL validate if it
   is strictly increasing and positive.
   \item Validate if input info0 is a numeric vector or NULL, if non-NULL validate if it
    is strictly increasing and positive.
   \item Validate if input info1 is a numeric vector or NULL, if non-NULL validate if it
   is strictly increasing and positive.
   \item Validate if input theta is a real vector and has the same length as info.
   \item Validate if input theta1 is a real vector and has the same length as info.
   \item Validate if input test_upper and test_lower are logical and have the same length as info.
   \item Validate if input test_upper value is TRUE.
   \item Validate if input alpha and beta are positive and of length one.
   \item Validate if input alpha and beta are from the unit interval and alpha is smaller than beta.
   \item Initialize bounds, numerical integration grids, boundary crossing probabilities.
   \item Compute fixed sample size for desired power and Type I error.
   \item Find an interval for information inflation to give correct power using \code{gs_power_npe()}.
   \item
   \item If there is no interim analysis, return a tibble including Analysis time, upper bound, Z-value,
   Probability of crossing bound, theta, info0 and info1.
   \item If the design is a group sequential design, return a tibble of Analysis,
    Bound, Z, Probability,  theta, info, info0.
  }
}
\if{html}{The contents of this section are shown in PDF user manual only.}
}

\examples{
library(dplyr)
library(gsDesign)

# ---------------------------------#
#         example 1                #
# ---------------------------------#
# Single analysis
# Lachin book p 71 difference of proportions example
pc <- .28 # Control response rate
pe <- .40 # Experimental response rate
p0 <- (pc + pe) / 2 # Ave response rate under H0

# Information per increment of 1 in sample size
info0 <- 1 / (p0 * (1 - p0) * 4)
info <- 1 / (pc * (1 - pc) * 2 + pe * (1 - pe) * 2)

# Result should round up to next even number = 652
# Divide information needed under H1 by information per patient added
gs_design_npe(theta = pe - pc, info = info, info0 = info0)


# ---------------------------------#
#         example 2                #
# ---------------------------------#
# Fixed bound
x <- gs_design_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 80,
  info0 = (1:3) * 80,
  upper = gs_b,
  upar = gsDesign::gsDesign(k = 3, sfu = gsDesign::sfLDOF)$upper$bound,
  lower = gs_b,
  lpar = c(-1, 0, 0)
)
x

# Same upper bound; this represents non-binding Type I error and will total 0.025
gs_power_npe(
  theta = rep(0, 3),
  info = (x \%>\% filter(bound == "upper"))$info,
  upper = gs_b,
  upar = (x \%>\% filter(bound == "upper"))$z,
  lower = gs_b,
  lpar = rep(-Inf, 3)
)

# ---------------------------------#
#         example 3                #
# ---------------------------------#
# Spending bound examples
# Design with futility only at analysis 1; efficacy only at analyses 2, 3
# Spending bound for efficacy; fixed bound for futility
# NOTE: test_upper and test_lower DO NOT WORK with gs_b; must explicitly make bounds infinite
# test_upper and test_lower DO WORK with gs_spending_bound
gs_design_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  info0 = (1:3) * 40,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_b,
  lpar = c(-1, -Inf, -Inf),
  test_upper = c(FALSE, TRUE, TRUE)
)

# one can try `info_scale = 1` or `info_scale = 0` here
gs_design_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  info0 = (1:3) * 30,
  info_scale = 1,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_b,
  lpar = c(-1, -Inf, -Inf),
  test_upper = c(FALSE, TRUE, TRUE)
)

# ---------------------------------#
#         example 4                #
# ---------------------------------#
# Spending function bounds
# 2-sided asymmetric bounds
# Lower spending based on non-zero effect
gs_design_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  info0 = (1:3) * 30,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfHSD, total_spend = 0.1, param = -1, timing = NULL)
)

# ---------------------------------#
#         example 5                #
# ---------------------------------#
# Two-sided symmetric spend, O'Brien-Fleming spending
# Typically, 2-sided bounds are binding
xx <- gs_design_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  binding = TRUE,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL)
)
xx

# Re-use these bounds under alternate hypothesis
# Always use binding = TRUE for power calculations
gs_power_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  binding = TRUE,
  upper = gs_b,
  lower = gs_b,
  upar = (xx \%>\% filter(bound == "upper"))$z,
  lpar = -(xx \%>\% filter(bound == "upper"))$z
)

}
\author{
Keaven Anderson \email{keaven_anderson@merck.com}
}
