% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_p.R
\name{add_p.tbl_summary}
\alias{add_p.tbl_summary}
\title{Adds p-values to summary tables}
\usage{
\method{add_p}{tbl_summary}(
  x,
  test = NULL,
  pvalue_fun = NULL,
  group = NULL,
  include = everything(),
  exclude = NULL,
  ...
)
}
\arguments{
\item{x}{Object with class \code{tbl_summary} from the \link{tbl_summary} function}

\item{test}{List of formulas specifying statistical tests to perform,
e.g. \code{list(all_continuous() ~ "t.test", all_categorical() ~ "fisher.test")}.
Options include
\itemize{
\item \code{"t.test"} for a t-test,
\item \code{"aov"} for a one-way ANOVA test,
\item \code{"wilcox.test"} for a Wilcoxon rank-sum test,
\item \code{"kruskal.test"} for a Kruskal-Wallis rank-sum test,
\item \code{"chisq.test"} for a chi-squared test of independence,
\item \code{"chisq.test.no.correct"} for a chi-squared test of independence without continuity correction,
\item \code{"fisher.test"} for a Fisher's exact test,
\item \code{"lme4"} for a random intercept logistic regression model to account for
clustered data, \code{lme4::glmer(by ~ variable + (1 | group), family = binomial)}.
The \code{by} argument must be binary for this option.
}

Tests default to \code{"kruskal.test"} for continuous variables, \code{"chisq.test"} for
categorical variables with all expected cell counts >=5, and \code{"fisher.test"}
for categorical variables with any expected cell count <5.
A custom test function can be added for all or some variables. See below for
an example.}

\item{pvalue_fun}{Function to round and format p-values.
Default is \link{style_pvalue}.
The function must have a numeric vector input (the numeric, exact p-value),
and return a string that is the rounded/formatted p-value (e.g.
\code{pvalue_fun = function(x) style_pvalue(x, digits = 2)} or equivalently,
\code{purrr::partial(style_pvalue, digits = 2)}).}

\item{group}{Column name (unquoted or quoted) of an ID or grouping variable.
The column can be used to calculate p-values with correlated data (e.g. when
the test argument is \code{"lme4"}). Default is \code{NULL}.  If specified,
the row associated with this variable is omitted from the summary table.}

\item{include}{Variables to include in output. Input may be a vector of
quoted variable names, unquoted variable names, or tidyselect select helper
functions. Default is \code{everything()}.}

\item{exclude}{DEPRECATED}

\item{...}{Not used}
}
\value{
A \code{tbl_summary} object
}
\description{
Adds p-values to tables created by \code{tbl_summary} by comparing values across groups.
}
\section{Setting Defaults}{

If you like to consistently use a different function to format p-values or
estimates, you can set options in the script or in the user- or
project-level start-up file, '.Rprofile'.  The default confidence level can
also be set. Please note the default option for the estimate is the same
as it is for \code{tbl_regression()}.
\itemize{
\item \code{options(gtsummary.pvalue_fun = new_function)}
}
}

\section{Example Output}{

\if{html}{Example 1}

\if{html}{\figure{add_p_ex1.png}{options: width=60\%}}

\if{html}{Example 2}

\if{html}{\figure{add_p_ex2.png}{options: width=60\%}}
}

\examples{
# Example 1 ----------------------------------
add_p_ex1 <-
  trial[c("age", "grade", "trt")] \%>\%
  tbl_summary(by = trt) \%>\%
  add_p()

# Example 2 ----------------------------------
# Conduct a custom McNemar test for response,
# Function must return a named list of the p-value and the
# test name: list(p = 0.123, test = "McNemar's test")
# The '...' must be included as input
# This feature is experimental, and the API may change in the future
my_mcnemar <- function(data, variable, by, ...) {
  result <- list()
  result$p <- stats::mcnemar.test(data[[variable]], data[[by]])$p.value
  result$test <- "McNemar's test"
  result
}

add_p_ex2 <-
  trial[c("response", "trt")] \%>\%
  tbl_summary(by = trt) \%>\%
  add_p(test = response ~ "my_mcnemar")
}
\seealso{
See tbl_summary \href{http://www.danieldsjoberg.com/gtsummary/articles/tbl_summary.html}{vignette} for detailed examples

Other tbl_summary tools: 
\code{\link{add_n.tbl_summary}()},
\code{\link{add_overall}()},
\code{\link{add_q}()},
\code{\link{add_stat_label}()},
\code{\link{bold_italicize_labels_levels}},
\code{\link{inline_text.tbl_summary}()},
\code{\link{inline_text.tbl_survfit}()},
\code{\link{modify}},
\code{\link{tbl_merge}()},
\code{\link{tbl_stack}()},
\code{\link{tbl_summary}()}
}
\author{
Emily C. Zabor, Daniel D. Sjoberg
}
\concept{tbl_summary tools}
