% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_wms_raster.R
\name{get_wms_raster}
\alias{get_wms_raster}
\title{Download WMS raster layer}
\usage{
get_wms_raster(x,
               layer = "ORTHOIMAGERY.ORTHOPHOTOS",
               res = 10,
               crs = 2154,
               rgb = TRUE,
               filename = tempfile(fileext = ".tif"),
               verbose = TRUE,
               overwrite = FALSE,
               interactive = FALSE)
}
\arguments{
\item{x}{Object of class \code{sf} or \code{sfc}, located in France.}

\item{layer}{\code{character}; layer name obtained from
\code{get_layers_metadata("wms-r")} or the
\href{https://geoservices.ign.fr/services-web-experts}{IGN website}.}

\item{res}{\code{numeric}; resolution specified in the units of
the coordinate system (e.g., meters for EPSG:2154, degrees for EPSG:4326).
See details for more information.}

\item{crs}{\code{numeric}, \code{character}, or object of class \code{sf} or \code{sfc};
defaults to EPSG:2154. See \code{\link[sf:st_crs]{sf::st_crs()}} for more details.}

\item{rgb}{\code{boolean}; if set to \code{TRUE}, downloads an RGB image. If set
to \code{FALSE}, downloads a single band with floating point values.
See details for more information.}

\item{filename}{\code{character} or \code{NULL}; specifies the filename or an
open connection for writing (e.g., "test.tif" or "~/test.tif").
If \code{NULL}, uses \code{layer} as the filename. The default format is ".tif",
but all \href{https://gdal.org/drivers/raster/index.html}{GDAL drivers}
are supported.}

\item{verbose}{\code{boolean}; if TRUE, message are added.}

\item{overwrite}{\code{boolean}; if TRUE, the existing raster will be overwritten.}

\item{interactive}{\code{logical}; if TRUE, an interactive menu prompts for
\code{apikey} and \code{layer}.}
}
\value{
\code{SpatRaster} object from \code{terra} package.
}
\description{
Download a raster layer from the IGN Web Mapping Services (WMS).
Specify a location using a shape and provide the layer name.
}
\details{
\itemize{
\item \code{res}: Note that setting \code{res} higher than the default resolution
of the layer will increase the number of pixels but not the precision
of the image. For instance, downloading the BD Alti layer from IGN
is optimal at a resolution of 25m.
\item \code{rgb}: Rasters are commonly used to download images such as orthophotos.
In specific cases like DEMs, however, a value per pixel is essential.
\item \code{overwrite}: The function \code{get_wms_raster} first checks if
\code{filename} already exists. If it does, the file is imported into
R without downloading again, unless \code{overwrite} is set to \code{TRUE}.
}
}
\examples{
\dontrun{
library(sf)
library(tmap)

# Shape from the best town in France
penmarch <- read_sf(system.file("extdata/penmarch.shp", package = "happign"))

# For quick testing use interactive = TRUE
raster <- get_wms_raster(x = penmarch, res = 25, interactive = TRUE)

# For specific data, choose apikey with get_apikey() and layer with get_layers_metadata()
apikey <- get_apikeys()[4]  # altimetrie
metadata_table <- get_layers_metadata("wms-r", apikey) # all layers for altimetrie wms
layer <- metadata_table[2,1] # ELEVATION.ELEVATIONGRIDCOVERAGE

# Downloading digital elevation model values not image
mnt_2154 <- get_wms_raster(penmarch, layer, res = 1, crs = 2154, rgb = FALSE)

# If crs is set to 4326, res is in degrees
mnt_4326 <- get_wms_raster(penmarch, layer, res = 0.0001, crs = 4326, rgb = FALSE)

# Plotting result
tm_shape(mnt_4326)+
   tm_raster()+
tm_shape(penmarch)+
   tm_borders(col = "blue", lwd  = 3)
}

}
\seealso{
\code{\link[=get_layers_metadata]{get_layers_metadata()}}
}
