% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/blueprint-recipe-default.R, R/mold.R
\name{default_recipe_blueprint}
\alias{default_recipe_blueprint}
\alias{mold.recipe}
\title{Default recipe blueprint}
\usage{
default_recipe_blueprint(
  intercept = FALSE,
  allow_novel_levels = FALSE,
  fresh = TRUE,
  bake_dependent_roles = character(),
  composition = "tibble"
)

\method{mold}{recipe}(x, data, ..., blueprint = NULL)
}
\arguments{
\item{intercept}{A logical. Should an intercept be included in the
processed data? This information is used by the \code{process} function
in the \code{mold} and \code{forge} function list.}

\item{allow_novel_levels}{A logical. Should novel factor levels be allowed at
prediction time? This information is used by the \code{clean} function in the
\code{forge} function list, and is passed on to \code{\link[=scream]{scream()}}.}

\item{fresh}{Should already trained operations be re-trained when \code{prep()} is
called?}

\item{bake_dependent_roles}{A character vector of recipes column "roles"
specifying roles that are required to \code{\link[recipes:bake]{recipes::bake()}} new data. Can't be
\code{"predictor"} or \code{"outcome"}, as predictors are always required and
outcomes are handled by the \code{outcomes} argument of \code{\link[=forge]{forge()}}.

Typically, non-standard roles (such as \code{"id"} or \code{"case_weights"}) are not
required to \code{bake()} new data. Unless specified by \code{bake_dependent_roles},
these non-standard role columns are excluded from checks done in \code{\link[=forge]{forge()}}
to validate the column structure of \code{new_data}, will not be passed to
\code{bake()} even if they existed in \code{new_data}, and will not be returned in
the \code{forge()$extras$roles} slot. See the documentation of
\code{\link[recipes:roles]{recipes::add_role()}} for more information about roles.}

\item{composition}{Either "tibble", "matrix", or "dgCMatrix" for the format
of the processed predictors. If "matrix" or "dgCMatrix" are chosen, all of
the predictors must be numeric after the preprocessing method has been
applied; otherwise an error is thrown.}

\item{x}{An unprepped recipe created from \code{\link[recipes:recipe]{recipes::recipe()}}.}

\item{data}{A data frame or matrix containing the outcomes and predictors.}

\item{...}{Not used.}

\item{blueprint}{A preprocessing \code{blueprint}. If left as \code{NULL}, then a
\code{\link[=default_recipe_blueprint]{default_recipe_blueprint()}} is used.}
}
\value{
For \code{default_recipe_blueprint()}, a recipe blueprint.
}
\description{
This pages holds the details for the recipe preprocessing blueprint. This
is the blueprint used by default from \code{mold()} if \code{x} is a recipe.
}
\section{Mold}{


When \code{mold()} is used with the default recipe blueprint:
\itemize{
\item It calls \code{\link[recipes:prep]{recipes::prep()}} to prep the recipe.
\item It calls \code{\link[recipes:juice]{recipes::juice()}} to extract the outcomes and predictors. These
are returned as tibbles.
\item If \code{intercept = TRUE}, adds an intercept column to the predictors.
}
}

\section{Forge}{


When \code{forge()} is used with the default recipe blueprint:
\itemize{
\item It calls \code{\link[=shrink]{shrink()}} to trim \code{new_data} to only the required columns and
coerce \code{new_data} to a tibble.
\item It calls \code{\link[=scream]{scream()}} to perform validation on the structure of the columns
of \code{new_data}.
\item It calls \code{\link[recipes:bake]{recipes::bake()}} on the \code{new_data} using the prepped recipe
used during training.
\item It adds an intercept column onto \code{new_data} if \code{intercept = TRUE}.
}
}

\examples{
library(recipes)

# ---------------------------------------------------------------------------
# Setup

train <- iris[1:100, ]
test <- iris[101:150, ]

# ---------------------------------------------------------------------------
# Recipes example

# Create a recipe that logs a predictor
rec <- recipe(Species ~ Sepal.Length + Sepal.Width, train) \%>\%
  step_log(Sepal.Length)

processed <- mold(rec, train)

# Sepal.Length has been logged
processed$predictors

processed$outcomes

# The underlying blueprint is a prepped recipe
processed$blueprint$recipe

# Call forge() with the blueprint and the test data
# to have it preprocess the test data in the same way
forge(test, processed$blueprint)

# Use `outcomes = TRUE` to also extract the preprocessed outcome!
# This logged the Sepal.Length column of `new_data`
forge(test, processed$blueprint, outcomes = TRUE)

# ---------------------------------------------------------------------------
# With an intercept

# You can add an intercept with `intercept = TRUE`
processed <- mold(rec, train, blueprint = default_recipe_blueprint(intercept = TRUE))

processed$predictors

# But you also could have used a recipe step
rec2 <- step_intercept(rec)

mold(rec2, iris)$predictors

# ---------------------------------------------------------------------------
# Non standard roles

# If you have custom recipes roles, it is assumed that they are required
# in `prep()` and afterwards for modeling, but are not required at `bake()`
# time and for prediction. This means that they are processed and returned
# in the `$extras$roles` slot of the return value of `mold()`, but they
# are not required to be in `new_data` in `forge()` and are not returned
# in the result.

rec_roles <- recipe(train) \%>\%
  update_role(Sepal.Width, new_role = "predictor") \%>\%
  update_role(Species, new_role = "outcome") \%>\%
  update_role(Sepal.Length, new_role = "id") \%>\%
  update_role(Petal.Length, new_role = "important")

processed_roles <- mold(rec_roles, train)

# The custom roles will still be in the `mold()` result in case you need
# them for modeling.
processed_roles$extras

# Notice that the columns with custom roles exist in `test`,
# but they weren't passed to `bake()` and aren't in the output.
forge(test, processed_roles$blueprint)$extras

# They can even be removed from `test` entirely, and it still works.
test2 <- test
test2$Petal.Length <- NULL
forge(test2, processed_roles$blueprint)$extras

# Occasionally, you might have a custom role that is required to be able
# to `bake()` on `new_data`. In those cases, you can specify it with
# `bake_dependent_roles` in `default_recipe_blueprint()`, which will ensure
# that it is a required column when calling `forge()`, that it will be
# passed on to `bake()`, and that it will be returned in the result.
bp <- default_recipe_blueprint(bake_dependent_roles = "important")

processed_roles <- mold(rec_roles, train, blueprint = bp)

# Now `"important"` is a required role when `forge()`-ing
forge(test, processed_roles$blueprint)$extras$roles

# Which means that we can't `forge()` with the data frame that is missing
# the `Petal.Length` column
try(forge(test2, processed_roles$blueprint))

# ---------------------------------------------------------------------------
# Matrix output for predictors

# You can change the `composition` of the predictor data set
bp <- default_recipe_blueprint(composition = "dgCMatrix")
processed <- mold(rec, train, blueprint = bp)
class(processed$predictors)
}
