% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{hpaSelection}
\alias{hpaSelection}
\title{Perform semi-nonparametric selection model estimation}
\usage{
hpaSelection(
  selection,
  outcome,
  data,
  z_K = 1L,
  y_K = 1L,
  pol_elements = 3L,
  is_Newey = FALSE,
  x0 = numeric(0),
  is_Newey_loocv = FALSE,
  cov_type = "sandwich",
  boot_iter = 100L,
  is_parallel = FALSE,
  opt_type = "optim",
  opt_control = NULL
)
}
\arguments{
\item{selection}{an object of class "formula" (or one that can be coerced to that class): 
a symbolic description of the selection equation form. All variables in \code{selection}
should be numeric vectors of the same length.}

\item{outcome}{an object of class "formula" (or one that can be coerced to that class): 
a symbolic description of the outcome equation form. All variables in \code{outcome}
should be numeric vectors of the same length.}

\item{data}{data frame containing the variables in the model.}

\item{z_K}{non-negative integer representing 
polynomial degree related to selection equation.}

\item{y_K}{non-negative integer representing polynomial 
degree related to outcome equation.}

\item{pol_elements}{number of conditional expectation approximating terms for Newey's method. If \code{is_Newey_loocv} is \code{TRUE}
then determines maximum number of these terms during leave-one-out cross-validation.}

\item{is_Newey}{logical; if TRUE then returns only Newey's method estimation results (default value is FALSE).}

\item{x0}{numeric vector of optimization routine initial values.
Note that \code{x0=c(pol_coefficients[-1], mean, sd, z_coef, y_coef)}.}

\item{is_Newey_loocv}{logical; if TRUE then number of conditional expectation approximating terms for Newey's method will be selected
based on leave-one-out cross-validation criteria iterating througt 0 to pol_elements number of these terms.}

\item{cov_type}{string value determining the type of covariance matrix to be
returned and used for summary. If \code{cov_type = "hessian"} then negative
inverse of Hessian matrix will be applied. If \code{cov_type = "gop"} then
inverse of Jacobian outer products will be used.
If \code{cov_type = "sandwich"} (default) then sandwich covariance matrix
estimator will be applied. If \code{cov_type = "bootstrap"} then bootstrap
with \code{boot_iter} iterations will be used.
If \code{cov_type = "hessianFD"} or \code{cov_type = "sandwichFD"} then
accurate but computationally demanding central difference Hessian
approximation will be calculated for the inverse Hessian and sandwich
estimators correspondingly. Central differences are computed via
analytically provided gradient. This Hessian matrix estimation approach
seems to be less accurate then BFGS approximation if polynomial order
is high (usually greater then 5).}

\item{boot_iter}{the number of bootstrap iterations
for \code{cov_type = "bootstrap"} covariance matrix estimator type.}

\item{is_parallel}{if TRUE then multiple cores will be
used for some calculations. It usually provides speed advantage for
large enough samples (about more than 1000 observations).}

\item{opt_type}{string value determining the type of the optimization
routine to be applied. The default is "optim" meaning that BFGS method
from the \code{\link[stats]{optim}} function will be applied.
If \code{opt_type = "GA"} then \code{\link[GA]{ga}} function will be
additionally applied.}

\item{opt_control}{a list containing arguments to be passed to the
optimization routine depending on \code{opt_type} argument value.
Please see details to get additional information.}
}
\value{
This function returns an object of class "hpaSelection".\cr \cr
An object of class "hpaSelection" is a list containing the following components:
\itemize{
\item \code{optim} - \code{\link[stats]{optim}} function output. 
If \code{opt_type = "GA"} then it is the list containing 
\code{\link[stats]{optim}} and \code{\link[GA]{ga}} functions outputs.
\item \code{x1} - numeric vector of distribution parameters estimates.
\item \code{Newey} - list containing information concerning Newey's method estimation results.
\item \code{z_mean} - estimate of the hermite polynomial mean parameter related to selection equation random error marginal distribution.
\item \code{y_mean} - estimate of the hermite polynomial mean parameter related to outcome equation random error marginal distribution.
\item \code{z_sd} - estimate of sd parameter related to selection equation random error marginal distribution.
\item \code{y_sd} - estimate of the hermite polynomial sd parameter related to outcome equation random error marginal distribution.
\item \code{pol_coefficients} - polynomial coefficients estimates.
\item \code{pol_degrees} - numeric vector which first element is \code{z_K} and the second is \code{y_K}.
\item \code{z_coef} - selection equation regression coefficients estimates.
\item \code{y_coef} - outcome equation regression coefficients estimates.
\item \code{cov_mat} - covariance matrix estimate.
\item \code{results} - numeric matrix representing estimation results.
\item \code{log-likelihood} - value of Log-Likelihood function.
\item \code{re_moments} - list which contains information about random errors expectations, variances and correlation.
\item \code{data_List} - list containing model variables and their partiotion according to outcome and selection equations.
\item \code{n_obs} - number of observations.
\item \code{ind_List} - list which contains information about parameters indexes in \code{x1}.
\item \code{selection_formula} - the same as \code{selection} input parameter.
\item \code{outcome_formula} - the same as \code{outcome} input parameter.}
Abovementioned list \code{Newey} has class "hpaNewey" and contains the following components:
\itemize{
\item \code{y_coef} - regression coefficients estimates (except constant term which is part of conditional expectation approximating polynomial).
\item \code{z_coef} - regression coefficients estimates related to selection equation.
\item \code{constant_biased} - biased estimate of constant term.
\item \code{inv_mills} - inverse mills rations estimates and their powers (including constant).
\item \code{inv_mills_coef} - coefficients related to \code{inv_mills}.
\item \code{pol_elements} - the same as \code{pol_elements} input parameter. However if \code{is_Newey_loocv} is \code{TRUE}
then it will equal to the number of conditional expectation approximating terms for Newey's method which
minimize leave-one-out cross-validation criteria.
\item \code{outcome_exp_cond} - dependend variable conditional expectation estimates.
\item \code{selection_exp} - selection equation random error expectation estimate.
\item \code{selection_var} - selection equation random error variance estimate.
\item \code{hpaBinaryModel} - object of class "hpaBinary" which contains selection equation estimation results.}
Abovementioned list \code{re_moments} contains the following components:
\itemize{
\item \code{selection_exp} - selection equation random errors expectation estimate.
\item \code{selection_var} - selection equation random errors variance estimate.
\item \code{outcome_exp} - outcome equation random errors expectation estimate.
\item \code{outcome_var} - outcome equation random errors variance estimate.
\item \code{errors_covariance} - outcome and selection equation random errors covariance estimate.
\item \code{rho} - outcome and selection equation random errors correlation estimate.
\item \code{rho_std} - outcome and selection equation random errors correlation estimator standard error estimate.}
}
\description{
This function performs semi-nonparametric selection model estimation
via hermite polynomial densities approximation.
}
\details{
Semi-nonparametric (SNP) approach has been implemented via
densities hermite polynomial approximation

Densities hermite polynomial approximation approach has been
proposed by A. Gallant and D. W. Nychka in 1987. The main idea is to
approximate unknown distribution density with hermite polynomial of  
degree \code{pol_degree}. In this framework hermite polynomial represents
adjusted (to insure integration to 1) product of squared polynomial and 
normal distribution densities. Parameters \code{mean} and \code{sd} 
determine means and standard deviations of normal distribution density
functions which are parts of this polynomial.
For more information please refer to the literature listed below.

Parameters \code{mean}, \code{sd}, \code{given_ind},
\code{omit_ind} should have the same length as \code{pol_degrees} 
parameter.

The first polynomial coefficient (zero powers) 
set to 1 for identification reasons.

Note that coefficient for the first
independent variable in \code{selection} will be fixed to 1.

All variables mentioned in \code{selection} and 
\code{outcome} should be numeric vectors.

The function calculates standard errors via sandwich estimator
and significance levels are reported taking into account quasi maximum
likelihood estimator (QMLE) asymptotic normality. If ones wants to switch
from QMLE to semi-nonparametric estimator (SNPE) during hypothesis testing
then covariance matrix should be estimated again using bootstrap.

Initial values for optimization routine are 
obtained by Newey's method (see the reference below).

Note that selection equation dependent variables should have exactly two levels (0 and 1) where "0" states for the selection results 
which leads to unobservable values of dependent variable in outcome equation.

This function maximizes (quasi) log-likelihood function 
via \code{\link[stats]{optim}} function setting it's \code{method} 
argument to "BFGS". If \code{opt_type = "GA"} then genetic
algorithm from \code{\link[GA]{ga}} function
will be additionally (after \code{\link[stats]{optim}} putting it's
solution (\code{par}) to \code{suggestions} matrix) applied in order to 
perform global optimization. Note that global optimization takes
much more time (usually minutes but sometimes hours or even days). 
The number of iterations and population size of the genetic algorithm
will grow linearly along with the number of estimated parameters. 
If it is seems that global maximum has not been found than it
is possible to continue the search restarting the function setting 
it's input argument \code{x0} to \code{x1} output value. Note that
if \code{cov_type = "bootstrap"} then \code{\link[GA]{ga}}
function will not be used for bootstrap iterations since it
may be extremely time consuming.

If \code{opt_type = "GA"} then \code{opt_control} should be the
list containing the values to be passed to \code{\link[GA]{ga}}
function. It is possible to pass arguments \code{lower}, \code{upper},
\code{popSize}, \code{pcrossover}, \code{pmutation}, \code{elitism},
\code{maxiter}, \code{suggestions}, \code{optim}, \code{optimArgs},
\code{seed} and \code{monitor}. 
Note that it is possible to set \code{population},
\code{selection}, \code{crossover} and \code{mutation} arguments changing
\code{\link[GA]{ga}} default parameters via \code{\link[GA]{gaControl}} 
function. These arguments information reported in \code{\link[GA]{ga}}.
In order to provide manual values for \code{lower} and \code{upper} bounds
please follow parameters ordering mentioned above for the
\code{x0} argument. If these bonds are not provided manually then
they (except those related to the polynomial coefficients)
will depend on the estimates obtained
by local optimization via \code{\link[stats]{optim}} function
(this estimates will be in the middle
between \code{lower} and \code{upper}).
Specifically for each sd parameter \code{lower} (\code{upper}) bound
is 5 times lower (higher) then this
parameter \code{\link[stats]{optim}} estimate.
For each mean and regression coefficient parameter it's lower and upper
bounds deviate from corresponding \code{\link[stats]{optim}} estimate
by two absolute value of this estimate.
Finally, lower and upper bounds for each polynomial
coefficient are \code{-10} and \code{10} correspondingly (do not depend
on their \code{\link[stats]{optim}} estimates).

The following arguments are differ from their defaults in
\code{\link[GA]{ga}}:
\itemize{
\item \code{pmutation = 0.2},
\item \code{optim = TRUE},
\item \code{optimArgs =
list("method" = "Nelder-Mead", "poptim" = 0.2, "pressel" = 0.5)},
\item \code{seed = 8},
\item \code{elitism = 2 + round(popSize * 0.1)}.}

Let's denote by \code{n_reg} the number of regressors
included to the \code{selection} and \code{outcome} formulas.
The arguments \code{popSize} and \code{maxiter} of
\code{\link[GA]{ga}} function have been set proportional to the number of
estimated polynomial coefficients and independent variables:
\itemize{
\item \code{popSize = 10 + 5 * (z_K + 1) * (y_K + 1) + 2 * n_reg}
\item \code{maxiter = 50 * (z_K + 1) * (y_K + 1) + 10 * n_reg}}
}
\examples{
\donttest{
## Let's estimate wage equation accounting for non-random selection.
## See the reference to Mroz TA (1987) to get additional details about
## the data this examples use

# Prepare data
library("sampleSelection")
data("Mroz87")
h = data.frame("kids" = as.numeric(Mroz87$kids5 + Mroz87$kids618 > 0),
	"age" = as.numeric(Mroz87$age),
	"faminc" = as.numeric(Mroz87$faminc),
	"educ" = as.numeric(Mroz87$educ),
	"exper" = as.numeric(Mroz87$exper),
	"city" = as.numeric(Mroz87$city),
	"wage" = as.numeric(Mroz87$wage),
	"lfp" = as.numeric(Mroz87$lfp))
	
# Estimate model parameters
model <- hpaSelection(selection = lfp ~ educ + age + I(age ^ 2) + 
                                  kids + log(faminc),
                      outcome = log(wage) ~ exper + I(exper ^ 2) + 
                                            educ + city,
                                  z_K = 2, y_K = 3, data = h, 
                                  pol_elements = 3, is_Newey_loocv = TRUE)
summary(model)

# Plot outcome equation random errorrs density
plot(model, is_outcome = TRUE)
# Plot selection equation random errorrs density
plot(model, is_outcome = FALSE)
}

## Estimate semi-nonparametric sample selection model
## parameters on simulated data given chi-squared random errors

\donttest{
set.seed(100)
library("mvtnorm")

# Sample size

n <- 1000

# Simulate independent variables
X_rho <- 0.5
X_sigma <- matrix(c(1,X_rho,X_rho,X_rho,1,X_rho,X_rho,X_rho,1), ncol=3)
X <- rmvnorm(n=n, mean = c(0,0,0), 
             sigma = X_sigma)

# Simulate random errors
epsilon <- matrix(0, n, 2)
epsilon_z_y <- rchisq(n, 5)
epsilon[, 1] <- (rchisq(n, 5) + epsilon_z_y) * (sqrt(3/20)) - 3.8736
epsilon[, 2] <- (rchisq(n, 5) + epsilon_z_y) * (sqrt(3/20)) - 3.8736
# Simulate selection equation
z_star <- 1 + 1 * X[,1] + 1 * X[,2] + epsilon[,1]
z <- as.numeric((z_star > 0))

# Simulate outcome equation
y_star <- 1 + 1 * X[,1] + 1 * X[,3] + epsilon[,2]
z <- as.numeric((z_star > 0))
y <- y_star
y[z==0] <- NA
h <- as.data.frame(cbind(z, y, X))
names(h) <- c("z", "y", "x1", "x2", "x3")

# Estimate parameters
model <- hpaSelection(selection = z ~ x1 + x2, 
                      outcome = y ~ x1 + x3,
                      data = h, z_K = 1, y_K = 3)
summary(model)

# Get conditional predictions for outcome equation
model_pred_c <- predict(model,is_cond = TRUE)
# Conditional predictions y|z=1
model_pred_c$y_1
# Conditional predictions y|z=0
model_pred_c$y_0

# Get unconditional predictions for outcome equation
model_pred_u <- predict(model,is_cond = FALSE)
model_pred_u$y

# Get conditional predictions for selection equation
# Note that for z=0 these predictions are NA
predict(model, is_cond = TRUE, is_outcome = FALSE)
# Get unconditional predictions for selection equation
predict(model, is_cond = FALSE, is_outcome = FALSE)
}

\dontshow{
set.seed(100)
library("mvtnorm")

# Sample size

n <- 100

# Simulate independent variables
X_rho <- 0.5
X_sigma <- matrix(c(1,X_rho,X_rho,X_rho,1,X_rho,X_rho,X_rho,1), ncol=3)
X <- rmvnorm(n=n, mean = c(0,0,0), 
             sigma = X_sigma)

# Simulate random errors
epsilon <- matrix(0, n, 2)
epsilon_z_y <- rchisq(n, 5)
epsilon[, 1] <- (rchisq(n, 5) + epsilon_z_y) * (sqrt(3/20)) - 3.8736
epsilon[, 2] <- (rchisq(n, 5) + epsilon_z_y) * (sqrt(3/20)) - 3.8736
# Simulate selection equation
z_star <- 1 + 1 * X[,1] + 1 * X[,2] + epsilon[,1]
z <- as.numeric((z_star > 0))

# Simulate outcome equation
y_star <- 1 + 1 * X[,1] + 1 * X[,3] + epsilon[,2]
z <- as.numeric((z_star > 0))
y <- y_star
y[z==0] <- NA
h <- as.data.frame(cbind(z, y, X))
names(h) <- c("z", "y", "x1", "x2", "x3")

# Estimate parameters
model <- hpaSelection(selection = z ~ x1 + x2, 
                      outcome = y ~ x1 + x3,
                      data = h, z_K = 1, y_K = 0)
summary(model)

# Get conditional predictions for outcome equation
model_pred_c <- predict(model,is_cond = TRUE)
# Conditional predictions y|z=1
model_pred_c$y_1
# Conditional predictions y|z=0
model_pred_c$y_0

# Get unconditional predictions for outcome equation
model_pred_u <- predict(model,is_cond = FALSE)
model_pred_u$y

# Get conditional predictions for selection equation
# Note that for z=0 these predictions are NA
predict(model, is_cond = TRUE, is_outcome = FALSE)
# Get unconditional predictions for selection equation
predict(model, is_cond = FALSE, is_outcome = FALSE)

# Plot outcome equation random errorrs density
plot(model)
}
}
\references{
A. Gallant and D. W. Nychka (1987) <doi:10.2307/1913241>

W. K. Newey (2009) <https://doi.org/10.1111/j.1368-423X.2008.00263.x>

Mroz T. A. (1987) <doi:10.2307/1911029>
}
\seealso{
\link[hpa]{summary.hpaSelection}, \link[hpa]{predict.hpaSelection}, \link[hpa]{plot.hpaSelection}, \link[hpa]{AIC.hpaSelection}, \link[hpa]{logLik.hpaSelection}
}
