#' Perform a request
#'
#' @description
#' After preparing a [request], call `req_perform()` to perform it, fetching
#' the results back to R as a [response].
#'
#' # Requests
#' Note that one call to `req_perform()` may perform multiple HTTP requests:
#'
#' * If the `url` is redirected with a 301, 302, 303, or 307, curl will
#'   automatically follow the `Location` header to the new location.
#'
#' * If you have configured retries with [req_retry()] and the request
#'   fails with a transient problem, `req_perform()` will try again after
#'   waiting a bit. See [req_retry()] for details.
#'
#' * If you are using OAuth, and the cached token has expired, `req_perform()`
#'   will get a new token either using the refresh token (if available)
#'   or by running the OAuth flow.
#'
#' @param req A [request].
#' @param path Optionally, path to save body of request. This is useful for
#'   large responses since it avoids storing the response in memory.
#' @param mock A mocking function. If supplied, this function is called
#'   with the request. It should return either `NULL` (if it doesn't want to
#'   handle the request) or a [response] (if it does). See [with_mock()]/
#'   `local_mock()` for more details.
#' @param verbosity How much information to print? This is a wrapper
#'   around `req_verbose()` that uses an integer to control verbosity:
#'
#'   * 0: no output
#'   * 1: show headers
#'   * 2: show headers and bodies
#'   * 3: show headers, bodies, and curl status messages.
#' @returns If request is successful (i.e. the request was successfully
#'   performed and a response with HTTP status code <400 was recieved), an HTTP
#'   [response]; otherwise throws an error. Override this behaviour with
#'   [req_error()].
#' @export
#' @examples
#' request("https://google.com") %>%
#'   req_perform()
req_perform <- function(
      req,
      path = NULL,
      verbosity = getOption("httr2_verbosity", 0L),
      mock = getOption("httr2_mock", NULL)
  ) {
  check_request(req)

  if (!is.null(mock)) {
    mock <- as_function(mock)
    mock_resp <- mock(req)
    if (!is.null(mock_resp)) {
      return(mock_resp)
    }
  }

  req <- req_verbosity(req, verbosity)
  req <- auth_oauth_sign(req)

  req <- cache_pre_fetch(req)
  if (is_response(req)) {
    return(req)
  }

  handle <- req_handle(req)
  max_tries <- retry_max_tries(req)
  deadline <- Sys.time() + retry_max_seconds(req)

  n <- 0
  tries <- 0
  delay <- throttle_delay(req)
  reauth <- FALSE # only ever re-authenticate once

  while(tries < max_tries && Sys.time() < deadline) {
    n <- n + 1

    sys_sleep(delay)
    resp <- tryCatch(
      req_perform1(req, path = path, handle = handle),
      error = function(err) {
        error_cnd("httr2_failed", message = conditionMessage(err), trace = trace_back())
      }
    )

    if (is_error(resp)) {
      tries <- tries + 1
      delay <- retry_backoff(req, tries)
    } else if (!reauth && resp_is_invalid_oauth_token(req, resp)) {
      reauth <- TRUE
      req <- auth_oauth_sign(req, TRUE)
      handle <- req_handle(req)
      delay <- 0
    } else if (retry_is_transient(req, resp)) {
      tries <- tries + 1
      delay <- retry_after(req, resp, tries)
    } else {
      # done
      break
    }
  }
  signal("", "httr2_fetch", n = n, tries = tries, reauth = reauth)

  resp <- cache_post_fetch(req, resp, path = path)

  if (is_error(resp)) {
    cnd_signal(resp)
  } else if (error_is_error(req, resp)) {
    resp_check_status(resp, error_body(req, resp))
  } else {
    resp
  }
}

req_perform1 <- function(req, path = NULL, handle = NULL) {
  the$last_request <- req
  the$last_response <- NULL

  if (!is.null(path)) {
    res <- curl::curl_fetch_disk(req$url, path, handle)
    body <- new_path(path)
  } else {
    res <- curl::curl_fetch_memory(req$url, handle)
    body <- res$content
  }

  resp <- new_response(
    method = req_method_get(req),
    url = res$url,
    status_code = res$status_code,
    headers = as_headers(res$headers),
    body = body
  )
  the$last_response <- resp
  resp
}

req_verbosity <- function(req, verbosity) {
  if (!is_integerish(verbosity, n = 1) || verbosity < 0 || verbosity > 3) {
    abort("verbosity must 0, 1, 2, or 3")
  }

  switch(verbosity + 1,
    req,
    req_verbose(req),
    req_verbose(req, body_req = TRUE, body_resp = TRUE),
    req_verbose(req, body_req = TRUE, body_resp = TRUE, info = TRUE)
  )
}

#' Retrieve most recent request/response
#'
#' These functions retrieve the most recent request made by httr2 and
#' the response it received, to facilitate debugging problems _after_ they
#' occur. If the request did not succeed (or no requests have been made)
#' `last_response()` will be `NULL`.
#'
#' @returns An HTTP [response]/[request].
#' @export
#' @examples
#' invisible(request("http://httr2.r-lib.org") %>% req_perform())
#' last_request()
#' last_response()
last_response <- function() {
  the$last_response
}

#' @export
#' @rdname last_response
last_request <- function() {
  the$last_request
}

#' Perform a dry run
#'
#' This shows you exactly what httr2 will send to the server, without
#' actually sending anything. It requires the httpuv package because it
#' works by sending the real HTTP request to a local webserver, thanks to
#' the magic of [curl::curl_echo()].
#'
#' @inheritParams req_verbose
#' @param quiet If `TRUE` doesn't print anything.
#' @returns Invisibly, a list containing information about the request,
#'   including `method`, `path`, and `headers`.
#' @export
#' @examples
#' # httr2 adds default User-Agent, Accept, and Accept-Encoding headers
#' request("http://example.com") %>% req_dry_run()
#'
#' # the Authorization header is automatically redacted to avoid leaking
#' # credentials on the console
#' req <- request("http://example.com") %>% req_auth_basic("user", "password")
#' req %>% req_dry_run()
#'
#' # if you need to see it, use redact_headers = FALSE
#' req %>% req_dry_run(redact_headers = FALSE)
req_dry_run <- function(req, quiet = FALSE, redact_headers = TRUE) {
  check_request(req)
  check_installed("httpuv")

  if (!quiet) {
    debug <- function(type, msg) {
      if (type == 2L) verbose_header("", msg, redact = redact_headers)
      if (type == 4L) verbose_message("", msg)
    }
    req <- req_options(req, debugfunction = debug, verbose = TRUE)
  }

  handle <- req_handle(req)
  curl::handle_setopt(handle, url = req$url)
  resp <- curl::curl_echo(handle, progress = FALSE)

  invisible(list(
    method = resp$method,
    path = resp$path,
    headers = as.list(resp$headers)
  ))
}

#' Perform a request, streaming data back to R
#'
#' After preparing a request, call `req_stream()` to perform the request
#' and handle the result with a streaming callback. This is useful for
#' streaming HTTP APIs where potentially the stream never ends.
#'
#' @inheritParams req_perform
#' @param callback A single argument callback function. It will be called
#'   repeatedly with a raw vector whenever there is at least `buffer_kb`
#'   worth of data to process. It must return `TRUE` to continue streaming.
#' @param timeout_sec Number of seconds to processs stream for.
#' @param buffer_kb Buffer size, in kilobytes.
#' @returns An HTTP [response].
#' @export
#' @examples
#' show_bytes <- function(x) {
#'   cat("Got ", length(x), " bytes\n", sep = "")
#'   TRUE
#' }
#' resp <- request("http://httpbin.org/stream-bytes/100000") %>%
#'   req_stream(show_bytes, buffer_kb = 32)
req_stream <- function(req, callback, timeout_sec = Inf, buffer_kb = 64) {
  check_request(req)

  handle <- req_handle(req)
  callback <- as_function(callback)

  stopifnot(is.numeric(timeout_sec), timeout_sec > 0)
  stop_time <- Sys.time() + timeout_sec

  stream <- curl::curl(req$url, handle = handle)
  open(stream, "rb")
  withr::defer(close(stream))

  continue <- TRUE
  while(continue && isIncomplete(stream) && Sys.time() < stop_time) {
    buf <- readBin(stream, raw(), buffer_kb * 1024)
    if (length(buf) > 0) {
      continue <- isTRUE(callback(buf))
    }
  }

  data <- curl::handle_data(handle)
  new_response(
    method = req_method_get(req),
    url = data$url,
    status_code = data$status_code,
    headers = as_headers(data$headers),
    body = NULL
  )
}

req_handle <- function(req) {
  req <- req_method_apply(req)
  req <- req_body_apply(req)

  if (!has_name(req$options, "useragent")) {
    req <- req_user_agent(req)
  }

  handle <- curl::new_handle()
  curl::handle_setheaders(handle, .list = headers_flatten(req$headers))
  curl::handle_setopt(handle, .list = req$options)
  if (length(req$fields) > 0) {
    curl::handle_setform(handle, .list = req$fields)
  }

  handle
}

new_path <- function(x) structure(x, class = "httr_path")
is_path <- function(x) inherits(x, "httr_path")
