% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimateCoeffs.R
\name{estimateCoeffs}
\alias{estimateCoeffs}
\alias{estimateInbreeding}
\alias{estimateTwoLocusInbreeding}
\alias{estimateKinship}
\alias{estimateTwoLocusKinship}
\alias{estimateKappa}
\alias{estimateTwoLocusKappa}
\alias{estimateIdentity}
\alias{estimateTwoLocusIdentity}
\title{Estimation of one- and two-locus relatedness coefficients}
\usage{
estimateInbreeding(x, id, Nsim, Xchrom = FALSE, verbose = FALSE, ...)

estimateTwoLocusInbreeding(
  x,
  id,
  rho = NULL,
  cM = NULL,
  Nsim,
  Xchrom = FALSE,
  verbose = FALSE,
  ...
)

estimateKinship(x, ids, Nsim, Xchrom = FALSE, verbose = FALSE, ...)

estimateTwoLocusKinship(
  x,
  ids,
  rho = NULL,
  cM = NULL,
  Nsim,
  Xchrom = FALSE,
  verbose = FALSE,
  ...
)

estimateKappa(x, ids, Nsim, Xchrom = FALSE, verbose = FALSE, ...)

estimateTwoLocusKappa(
  x,
  ids,
  rho = NULL,
  cM = NULL,
  Nsim,
  Xchrom = FALSE,
  verbose = FALSE,
  ...
)

estimateIdentity(x, ids, Nsim, Xchrom = FALSE, verbose = FALSE, ...)

estimateTwoLocusIdentity(
  x,
  ids,
  rho = NULL,
  cM = NULL,
  Nsim,
  Xchrom = FALSE,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{x}{A pedigree in the form of a \code{\link[pedtools:ped]{pedtools::ped()}} object.}

\item{id, ids}{A vector of one or two ID labels.}

\item{Nsim}{The number of simulations.}

\item{Xchrom}{A logical indicating if the loci are X-linked (if TRUE) or
autosomal (FALSE).}

\item{verbose}{A logical.}

\item{...}{Further arguments passed on to \code{\link[=ibdsim]{ibdsim()}}, e.g. \code{seed}.}

\item{rho}{A scalar in the interval \verb{[0, 0.5]}: the recombination fraction
between the two loci, converted to centiMorgans using Haldane's map function:
cM = -50 * log(1 - 2 * rho). Either \code{rho} or \code{cM} (but not both) must be
non-NULL.}

\item{cM}{A non-negative number: the genetic distance between the two loci,
given in centiMorgans. Either \code{rho} or \code{cM} (but not both) must be
non-NULL.}
}
\value{
\code{estimateInbreeding()}: a single probability.

\code{estimateTwoLocusInbreeding()}: a single probability.

\code{estimateKappa()}: a numeric vector of length 3, with the estimated
\eqn{\kappa} coefficients.

\code{estimateTwoLocusKappa()}: a symmetric, numerical 3*3 matrix, with the
estimated values of \eqn{\kappa_{ij}}, for \eqn{i,j = 0,1,2}.

\code{estimateIdentity()}: a numeric vector of length 9, with the estimated
identity coefficients.

\code{estimateTwoLocusIdentity()}: a symmetric, numerical 9*9 matrix, with the
estimated values of \eqn{\Delta_{ij}}, for \eqn{i,j = 1,...,9}.
}
\description{
Estimate by simulation various relatedness coefficients, and two-locus
versions of the same coefficients, for a given recombination rate. The
current implementation covers inbreeding coefficients, kinship coefficients,
IBD (kappa) coefficients between noninbred individuals, and condensed
identity coefficients. These functions are primarily meant as tools for
validating exact algorithms, e.g., as implemented in the \code{ribd} package.
}
\details{
In the following, let L1 and L2 denote two arbitrary autosomal loci with
recombination rate \eqn{\rho}, and let A and B be members of the pedigree
\code{x}.

The \emph{two-locus inbreeding coefficient} \eqn{f_2(\rho)} of A is defined as the
probability that A is autozygous at both L1 and L2 simultaneously.

The \emph{two-locus kinship coefficient} \eqn{\phi_2(\rho)} of A and B is defined
as the probability that a random gamete emitted from A, and a random gamete
emitted from B, contain IBD alleles at both L1 and L2.

The \emph{two-locus kappa coefficient} \eqn{\kappa_{ij}(\rho)}, for \eqn{i,j =
0,1,2}, of noninbred A and B, is the probability that A and B share exactly
\code{i} alleles IBD at L1, and exactly \code{j} alleles IBD at L2.

The \emph{two-locus identity coefficient} \eqn{\Delta_{ij}}, \eqn{i,j = 1,...,9}
is defined for any (possibly inbred) A and B, as the probability that A and B
are in identity state \code{i} at L1, and state \code{j} at L2. This uses the
conventional ordering of the nine condensed identity states. For details, see
for instance the \href{https://github.com/magnusdv/ribd}{GitHub page of the \code{ribd} package}.
}
\examples{

############################
### Two-locus inbreeding ###
############################

x = cousinPed(0, child = TRUE)
rho = 0.25
Nsim = 10 # Increase!
estimateTwoLocusInbreeding(x, id = 5, rho = rho, Nsim = Nsim, seed = 123)

########################################
### Two-locus kappa:                 ###
### Grandparent vs half sib vs uncle ###
########################################

# These are indistinguishable with unlinked loci, see e.g.
# pages 182-183 in Egeland, Kling and Mostad (2016).
# In the following, each simulation approximation is followed
# by its exact counterpart.

rho = 0.25; R = .5 * (rho^2 + (1-rho)^2)
Nsim = 10 # Should be increased to at least 10000

# Grandparent/grandchild
G = linearPed(2); G.ids = c(1,5); # plot(G, hatched = G.ids)
estimateTwoLocusKappa(G, G.ids, rho = rho, Nsim = Nsim, seed = 123)[2,2]
.5*(1-rho) # exact

# Half sibs
H = halfSibPed(); H.ids = c(4,5); # plot(H, hatched = H.ids)
estimateTwoLocusKappa(H, H.ids, rho = rho, Nsim = Nsim, seed = 123)[2,2]
R # exact

# Uncle
U = cousinPed(0, removal = 1); U.ids = c(3,6); # plot(U, hatched = U.ids)
estimateTwoLocusKappa(U, U.ids, rho = rho, Nsim = Nsim, seed = 123)[2,2]
(1-rho) * R + rho/4 # exact

# Exact calculations by ribd:
# ribd::twoLocusIBD(G, G.ids, rho = rho, coefs = "k11")
# ribd::twoLocusIBD(H, H.ids, rho = rho, coefs = "k11")
# ribd::twoLocusIBD(U, U.ids, rho = rho, coefs = "k11")

##########################
### Two-locus Jacquard ###
##########################

x = fullSibMating(1)
rho = 0.25
Nsim = 10 # (increase to at least 10000)

estimateTwoLocusIdentity(x, ids = 5:6, rho = rho, Nsim = Nsim, seed = 123)

# Exact by ribd:
# ribd::twoLocusIdentity(x, ids = 5:6, rho = rho)

}
