% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Manningpara.R
\encoding{UTF-8}
\name{Manningpara}
\alias{Manningpara}
\title{Parabolic cross-section for the Gauckler-Manning-Strickler equation}
\source{
r - Better error message for stopifnot? - Stack Overflow answered by Andrie on Dec 1 2011. See \url{https://stackoverflow.com/questions/8343509/better-error-message-for-stopifnot}.
}
\usage{
Manningpara(
  Q = NULL,
  n = NULL,
  m = NULL,
  Sf = NULL,
  y = NULL,
  B1 = NULL,
  y1 = NULL,
  T = NULL,
  units = c("SI", "Eng")
)
}
\arguments{
\item{Q}{numeric vector that contains the discharge value [m^3/s or ft^3/s],
if known.}

\item{n}{numeric vector that contains the Manning's roughness coefficient n,
if known.}

\item{m}{numeric vector that contains the "cross-sectional side slope of m:1
(horizontal:vertical)", if known.}

\item{Sf}{numeric vector that contains the bed slope (m/m or ft/ft),
if known.}

\item{y}{numeric vector that contains the flow depth (m or ft), if known.}

\item{B1}{numeric vector that contains the "bank-full width", if known.}

\item{y1}{numeric vector that contains the "bank-full depth", if known.}

\item{T}{numeric vector that contains the temperature (degrees C or degrees
Fahrenheit), if known.}

\item{units}{character vector that contains the system of units [options are
\code{SI} for International System of Units and \code{Eng} for English units
(United States Customary System in the United States and Imperial Units in
the United Kingdom)]}
}
\value{
the missing parameter (Q, n, m, Sf, B1, y1, or y) & area (A), wetted
  perimeter (P), velocity (V), top width (B), hydraulic radius (R),
  Reynolds number (Re), and Froude number (Fr) as a \code{\link[base]{list}}.
}
\description{
This function solves for one missing variable in the Gauckler-Manning-
Strickler equation for a parabolic cross-section and uniform flow. The
\code{\link[stats]{uniroot}} function is used to obtain the missing parameter.
}
\details{
Gauckler-Manning-Strickler equation is expressed as

\deqn{V = \frac{K_n}{n}R^\frac{2}{3}S^\frac{1}{2}}

\describe{
\item{\emph{V}}{the velocity (m/s or ft/s)}
\item{\emph{n}}{Manning's roughness coefficient (dimensionless)}
\item{\emph{R}}{the hydraulic radius (m or ft)}
\item{\emph{S}}{the slope of the channel bed (m/m or ft/ft)}
\item{\emph{\eqn{K_n}}}{the conversion constant -- 1.0 for SI and
       3.2808399 ^ (1 / 3) for English units -- m^(1/3)/s or ft^(1/3)/s}
}




This equation is also expressed as

\deqn{Q = \frac{K_n}{n}\frac{A^\frac{5}{3}}{P^\frac{2}{3}}S^\frac{1}{2}}

\describe{
\item{\emph{Q}}{the discharge [m^3/s or ft^3/s (cfs)] is VA}
\item{\emph{n}}{Manning's roughness coefficient (dimensionless)}
\item{\emph{P}}{the wetted perimeter of the channel (m or ft)}
\item{\emph{A}}{the cross-sectional area (m^2 or ft^2)}
\item{\emph{S}}{the slope of the channel bed (m/m or ft/ft)}
\item{\emph{\eqn{K_n}}}{the conversion constant -- 1.0 for SI and
       3.2808399 ^ (1 / 3) for English units -- m^(1/3)/s or ft^(1/3)/s}
}




Other important equations regarding the parabolic cross-section follow:
\deqn{R = \frac{A}{P}}

\describe{
\item{\emph{R}}{the hydraulic radius (m or ft)}
\item{\emph{A}}{the cross-sectional area (m^2 or ft^2)}
\item{\emph{P}}{the wetted perimeter of the channel (m or ft)}
}


\deqn{A = \frac{2}{3}By}

\describe{
\item{\emph{A}}{the cross-sectional area (m^2 or ft^2)}
\item{\emph{y}}{the flow depth (normal depth in this function) [m or ft]}
\item{\emph{B}}{the top width of the channel (m or ft)}
}



\deqn{P = \left(\frac{B}{2}\right)\left[\sqrt{\left(1 + x^2\right)} + \left(\frac{1}{x}\right) \ln \left(x + \sqrt{\left(1 + x^2\right)}\right)\right]}

\describe{
\item{\emph{P}}{the wetted perimeter of the channel (m or ft)}
\item{\emph{B}}{the top width of the channel (m or ft)}
\item{\emph{x}}{4y/b (dimensionless)}
}




\deqn{x = \frac{4y}{B}}

\describe{
\item{\emph{x}}{4y/b (dimensionless)}
\item{\emph{B}}{the top width of the channel (m or ft)}
\item{\emph{y}}{the flow depth (normal depth in this function) [m or ft]}
}




\deqn{B = B_1 \left(\sqrt{\frac{y}{y_1}}\right)}

\describe{
\item{\emph{B}}{the top width of the channel (m or ft)}
\item{\emph{y}}{the flow depth (normal depth in this function) [m or ft]}
\item{\emph{\eqn{B_1}}}{the "bank-full width" (m or ft)}
\item{\emph{\eqn{y_1}}}{the "bank-full depth" (m or ft)}
}



\deqn{D = \frac{A}{B}}

\describe{
\item{\emph{D}}{the hydraulic depth (m or ft)}
\item{\emph{A}}{the cross-sectional area (m^2 or ft^2)}
\item{\emph{B}}{the top width of the channel (m or ft)}
}


A rough turbulent zone check is performed on the water flowing in the
channel using the Reynolds number (Re). The Re equation follows:

\deqn{Re = \frac{\rho RV}{\mu}}

\describe{
\item{\emph{Re}}{Reynolds number (dimensionless)}
\item{\emph{\eqn{\rho}}}{density (kg/m^3 or slug/ft^3)}
\item{\emph{R}}{the hydraulic radius (m or ft)}
\item{\emph{V}}{the velocity (m/s or ft/s)}
\item{\emph{\eqn{\mu}}}{dynamic viscosity (* 10^-3 kg/m*s or * 10^-5 lb*s/ft^2)}
}



A critical flow check is performed on the water flowing in the channel
using the Froude number (Fr). The Fr equation follows:

\deqn{Fr = \frac{V}{\left(\sqrt{g * D}\right)}}

\describe{
\item{\emph{Fr}}{the Froude number (dimensionless)}
\item{\emph{V}}{the velocity (m/s or ft/s)}
\item{\emph{g}}{gravitational acceleration (m/s^2 or ft/sec^2)}
\item{\emph{D}}{the hydraulic depth (m or ft)}
}
}
\note{
Assumptions: uniform flow, prismatic channel, and surface water temperature
of 20 degrees Celsius (68 degrees Fahrenheit) at atmospheric pressure

Note: Units must be consistent
}
\examples{
library("iemisc")
# Exercise 4.3 from Sturm (page 153)
y <- Manningpara(Q = 12.0, B1 = 10, y1 = 2.0, Sf = 0.005, n = 0.05, units = "SI")
# defines all list values within the object named y
# Q = 12.0 m^3/s, B1 = 10 m, y1 = 2.0 m, Sf = 0.005 m/m, n = 0.05, units = SI units
# This will solve for y since it is missing and y will be in m

y$y # gives the value of y


# Modified Exercise 4.3 from Sturm (page 153)
Manningpara(y = y$y, B1 = 10, y1 = 2.0, Sf = 0.005, n = 0.05, units = "SI")
# y = 1.254427 m, B1 = 10 m, y1 = 2.0 m, Sf = 0.005 m/m, n = 0.05, units = SI units
# This will solve for Q since it is missing and Q will be in m^3/s


}
\references{
\enumerate{
   \item Terry W. Sturm, \emph{Open Channel Hydraulics}, 2nd Edition, New York City, New York: The McGraw-Hill Companies, Inc., 2010, page 2, 8, 36, 102, 120, 153.
   \item Dan Moore, P.E., NRCS Water Quality and Quantity Technology Development Team, Portland Oregon, "Using Mannings Equation with Natural Streams", August 2011, \url{http://www.wcc.nrcs.usda.gov/ftpref/wntsc/H&H/xsec/manningsNaturally.pdf}.
   \item Gilberto E. Urroz, Utah State University Civil and Environmental Engineering - OCW, CEE6510 - Numerical Methods in Civil Engineering, Spring 2006 (2006). Course 3. "Solving selected equations and systems of equations in hydraulics using Matlab", August/September 2004, \url{https://digitalcommons.usu.edu/ocw_cee/3}.
   \item Tyler G. Hicks, P.E., \emph{Civil Engineering Formulas: Pocket Guide}, 2nd Edition, New York City, New York: The McGraw-Hill Companies, Inc., 2002, page 423, 425.
   \item Wikimedia Foundation, Inc. Wikipedia, 26 November 2015, “Manning formula”, \url{https://en.wikipedia.org/wiki/Manning_formula}.
   \item John C. Crittenden, R. Rhodes Trussell, David W. Hand, Kerry J. Howe, George Tchobanoglous, \emph{MWH's Water Treatment: Principles and Design}, Third Edition, Hoboken, New Jersey: John Wiley & Sons, Inc., 2012, page 1861-1862.
   \item Andrew Chadwick, John Morfett and Martin Borthwick, \emph{Hydraulics in Civil and Environmental Engineering}, Fourth Edition, New York City, New York: Spon Press, Inc., 2004, page 133.
   \item Robert L. Mott and Joseph A. Untener, \emph{Applied Fluid Mechanics}, Seventh Edition, New York City, New York: Pearson, 2015, page 376.
   \item Wikimedia Foundation, Inc. Wikipedia, 17 March 2017, “Gravitational acceleration”, \url{https://en.wikipedia.org/wiki/Gravitational_acceleration}.
   \item Wikimedia Foundation, Inc. Wikipedia, 29 May 2016, “Conversion of units”, \url{https://en.wikipedia.org/wiki/Conversion_of_units}.
}
}
\seealso{
\code{\link{Manningtrap}} for a trapezoidal cross-section, \code{\link{Manningrect}} for a
  rectangular cross-section, \code{\link{Manningtri}} for a triangular cross-section,
  and \code{\link{Manningcirc}} for a circular cross-section.
}
