% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FeatureEffects.R
\name{FeatureEffects}
\alias{FeatureEffects}
\title{Effect of all features on predictions}
\format{\code{\link{R6Class}} object.}
\description{
\code{FeatureEffects} computes feature effects for multiple features at once.
}
\section{Usage}{

\preformatted{
effects = FeatureEffects$new(predictor, features = NULL, method = "ale", 
    grid.size = 20, center.at = NULL, parallel = FALSE)

plot(effects)
effects$results
print(effects)
}
}

\section{Arguments}{


For FeatureEffects$new():
\describe{
\item{predictor: }{(Predictor)\cr 
The object (created with Predictor$new()) holding the machine learning model and the data.}
\item{features: }{(`character()`)\cr 
The names of the features for which the effects should be computed.
Default is all features used in the prediction model.}
\item{method: }{(`character(1)`)\cr 
'ale' for accumulated local effects (the default), 
'pdp' for partial dependence plot, 
'ice' for individual conditional expectation curves,
'pdp+ice' for partial dependence plot and ice curves within the same plot.}
\item{center.at: }{(`numeric(1)`)\cr Value at which the plot should be centered.}
\item{grid.size: }{(`numeric(1)` | `numeric(2)`)\cr The size of the grid for evaluating the predictions}
\item{parallel: }{`logical(1)`\cr Should the method be executed in parallel? If TRUE, requires a cluster to be registered, see ?foreach::foreach.}
}
}

\section{Details}{


FeatureEffects computes the effects for all given features on the model prediction.
FeatureEffects is a convenience class that calls FeatureEffect multiple times.
See ?FeatureEffect for details what's actually computed.

Only first-order effects can be computed with the FeatureEffects interface. 
If you are intereated in the visualization of interactions between two features, directly use FeatureEffect.
}

\section{Fields}{

\describe{
\item{method: }{(`character(1)`)\cr
'ale' for accumulated local effects, 
'pdp' for partial dependence plot, 
'ice' for individual conditional expectation curves,
'pdp+ice' for partial dependence plot and ice curves within the same plot.}
\item{features: }{(`character()`)\cr
The names of the features for which the effects were computed.}
\item{grid.size: }{(`numeric(1)` | `numeric(2)`)\cr
The size of the grid.}
\item{center.at: }{(`numeric(1)` | `character(1)`)\cr
The value for the centering of the plot. Numeric for numeric features, and the level name for factors.}
\item{predictor: }{(Predictor)\cr
The prediction model that was analysed.}
\item{results: }{(list)\cr
A list with the results of each feature effect. Each entry is a data.frame with the grid of feature of interest and the predicted \eqn{\hat{y}}. 
Can be used for creating custom effect plots.}
\item{effects: }{(list)\cr
A list of the FeatureEffect objects for each feature. See ?FeatureEffect what you can do with them (e.g. plot them individually).
}
}
}

\section{Methods}{

\describe{
\item{plot()}{method to plot the all effects. See \link{plot.FeatureEffects}}
\item{\code{clone()}}{[internal] method to clone the R6 object.}
\item{\code{initialize()}}{[internal] method to initialize the R6 object.}
}
}

\examples{
# We train a random forest on the Boston dataset:
if (require("rpart")) {
data("Boston", package  = "MASS")
rf = rpart(medv ~ ., data = Boston)
mod = Predictor$new(rf, data = Boston)

# Compute the accumulated local effects for all features
eff = FeatureEffects$new(mod)
eff$plot()

\dontrun{
# Again, but this time with a partial dependence plot
eff = FeatureEffects$new(mod, method = "pdp")
eff$plot()

# Only a subset of features
eff = FeatureEffects$new(mod, features = c("nox", "crim"))
eff$plot()


# You can access each FeatureEffect individually

eff.nox = eff$effects[["nox"]]
eff.nox$plot()


# FeatureEffects also works with multiclass classification
rf = rpart(Species ~ ., data = iris)
mod = Predictor$new(rf, data = iris, type = "prob")

FeatureEffects$new(mod)$plot(ncol = 2)
}
}
}
\references{
Apley, D. W. 2016. "Visualizing the Effects of Predictor Variables in Black Box Supervised Learning Models." ArXiv Preprint.

Friedman, J.H. 2001. "Greedy Function Approximation: A Gradient Boosting Machine." Annals of Statistics 29: 1189-1232.

Goldstein, A., Kapelner, A., Bleich, J., and Pitkin, E. (2013). Peeking Inside the Black Box: 
Visualizing Statistical Learning with Plots of Individual Conditional Expectation, 1-22. https://doi.org/10.1080/10618600.2014.907095
}
\seealso{
\link{plot.FeatureEffect}
}
