#' Surface-Water Measurements
#'
#' @description Stage measurements and calculated discharge at gauging stations in the
#'   U.S. Geological Survey (USGS) surface-water monitoring network,
#'   Idaho National Laboratory and vicinity, Idaho.
#'   The purpose of this network is to provide information on the availability
#'   and distribution of surface water resources.
#'
#' @format A data frame with columns:
#' \describe{
#'   \item{`site_nm`}{Local site name.}
#'   \item{`site_no`}{USGS site number.}
#'   \item{`stage_dt`}{Date and time the water level was measured, in "America/Denver" time zone.
#'     Missing values of time were substituted with "12:00".}
#'   \item{`stage_va`}{Gage height (stage) as shown on
#'     the inside staff gage at the site or read off the recorder inside the gage house in feet.}
#'   \item{`disch_va`}{Computed discharge in cubic feet per second (cfs).}
#'   \item{`stage_acy_va`}{Accuracy of stage measurment in feet.}
#'   \item{`disch_acy_va`}{Accuracy of computed discharge in cfs.}
#' }
#'
#' @source Data was obtained from the National Water Information System (NWIS) (U.S. Geological Survey, 2023).
#'   Groundwater level data was retrieved from NWIS by using the USGS R-package
#'   [dataRetrieval](https://code.usgs.gov/water/dataRetrieval) (DeCicco and others, 2023).
#'   Data was retrieved for each site number listed in the samples dataset and removed any non-essential columns.
#'
#' @references DeCicco, L.A., Hirsch, R.M., Lorenz, D., Watkins, W.D., Johnson, M., 2023,
#'   dataRetrieval: R packages for discovering and retrieving water data available from
#'   Federal hydrologic web services, v.2.7.13, \doi{10.5066/P9X4L3GE}.
#' @references U.S. Geological Survey, 2023, National Water Information System---web services,
#'   accessed January 10, 2024, from \doi{10.5066/F7P55KJN}.
#'
#' @keywords datasets
#'
#' @examples
#' str(swm)
#'
#' poi <- as.POSIXct(c("2000-01-01", "2024-01-01")) # period of interest
#' site_no <- "13132500" # station BIG LOST RIVER NR ARCO ID
#' is <- swm$stage_dt >= poi[1] & swm$stage_dt < poi[2] & swm$site_no == site_no
#' d <- swm[is, ]
#' plotrix::plotCI(
#'   x = d$stage_dt,
#'   y = d$stage_va,
#'   li = d$stage_va - d$stage_acy_va,
#'   ui = d$stage_va + d$stage_acy_va
#' )
"swm"
