#' Build Package Vignettes
#'
#' Build package vignettes from their sources and place in the \code{/inst/doc} folder.
#'
#' @param pkg 'character' string.
#'   Package path, by default the \link[=getwd]{working directory}.
#' @param quiet 'logical' flag.
#'   Whether to supress most output.
#' @param clean 'logical' flag.
#'   Whether to remove all intermediate files generated by the build.
#' @param gs_quality 'character' string.
#'   Quality of compacted PDF files,
#'   see \code{\link[tools]{compactPDF}} function for details.
#'
#' @author J.C. Fisher, U.S. Geological Survey, Idaho Water Science Center
#'
#' @seealso \code{\link[tools]{buildVignettes}}
#'
#' @keywords utilities
#'
#' @export
#'
#' @examples
#' \dontrun{
#' BuildVignettes("<path/to/package>", gs_quality = "ebook")
#' }
#'

BuildVignettes <- function(pkg=".", quiet=TRUE, clean=TRUE, gs_quality=NULL) {

  checkmate::assertFileExists(file.path(pkg, "DESCRIPTION"))
  checkmate::assertFlag(quiet)
  checkmate::assertFlag(clean)
  if (!is.null(gs_quality))
    gs_quality <- match.arg(gs_quality, c("none", "printer", "ebook", "screen"))

  tools::buildVignettes(dir=pkg, quiet=quiet, clean=clean, tangle=TRUE)

  v <- tools::pkgVignettes(dir=pkg, output=TRUE, source=TRUE)
  if (length(v) == 0) return(invisible(NULL))
  out <- c(v$outputs, unique(unlist(v$sources, use.names=FALSE)))

  doc <- file.path(pkg, "inst/doc")

  dir.create(doc, showWarnings=!quiet, recursive=TRUE)
  file.copy(c(v$docs, out), doc, overwrite=TRUE)
  file.remove(out)

  if (!is.null(gs_quality))
    tools::compactPDF(paths=doc, gs_quality=gs_quality)

  invisible(TRUE)
}
