% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ConnectionWeights.R
\name{ConnectionWeights}
\alias{ConnectionWeights}
\title{Connection Weights Method}
\description{
This class implements the \emph{Connection Weights} method investigated by
Olden et al. (2004) which results in a feature relevance score for each input
variable. The basic idea is to multiply up all path weights for each
possible connection between an input feature and the output node and then
calculate the sum over them. Besides, it is a global interpretation method
and independent of the input data. For a neural network with \eqn{3} hidden
layers with weight matrices \eqn{W_1}, \eqn{W_2} and \eqn{W_3} this method
results in a simple matrix multiplication
\deqn{W_1 * W_2 * W_3. }
}
\examples{
\dontshow{if (torch::torch_is_installed()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
#----------------------- Example 1: Torch ----------------------------------
library(torch)

# Create nn_sequential model
model <- nn_sequential(
  nn_linear(5, 12),
  nn_relu(),
  nn_linear(12, 1),
  nn_sigmoid()
)

# Create Converter with input names
converter <- Converter$new(model,
  input_dim = c(5),
  input_names = list(c("Car", "Cat", "Dog", "Plane", "Horse"))
)

# Apply method Connection Weights
cw <- ConnectionWeights$new(converter)

# Print the result as a data.frame
cw$get_result("data.frame")

# Plot the result
plot(cw)

#----------------------- Example 2: Neuralnet ------------------------------
library(neuralnet)
data(iris)

# Train a Neural Network
nn <- neuralnet((Species == "setosa") ~ Petal.Length + Petal.Width,
  iris,
  linear.output = FALSE,
  hidden = c(3, 2), act.fct = "tanh", rep = 1
)

# Convert the trained model
converter <- Converter$new(nn)

# Apply the Connection Weights method
cw <- ConnectionWeights$new(converter)

# Get the result as a torch tensor
cw$get_result(type = "torch.tensor")

# Plot the result
plot(cw)

#----------------------- Example 3: Keras ----------------------------------
library(keras)

if (is_keras_available()) {
  # Define a model
  model <- keras_model_sequential()
  model \%>\%
    layer_conv_1d(
      input_shape = c(64, 3), kernel_size = 16, filters = 8,
      activation = "softplus"
    ) \%>\%
    layer_conv_1d(kernel_size = 16, filters = 4, activation = "tanh") \%>\%
    layer_conv_1d(kernel_size = 16, filters = 2, activation = "relu") \%>\%
    layer_flatten() \%>\%
    layer_dense(units = 64, activation = "relu") \%>\%
    layer_dense(units = 2, activation = "softmax")

  # Convert the model
  converter <- Converter$new(model)

  # Apply the Connection Weights method
  cw <- ConnectionWeights$new(converter)

  # Get the result as data.frame
  cw$get_result(type = "data.frame")

  # Plot the result for all classes
  plot(cw, output_idx = 1:2)
}

# ------------------------- Advanced: Plotly -------------------------------
# If you want to create an interactive plot of your results with custom
# changes, you can take use of the method plotly::ggplotly
library(ggplot2)
library(plotly)
library(neuralnet)
data(iris)

nn <- neuralnet(Species ~ .,
  iris,
  linear.output = FALSE,
  hidden = c(10, 8), act.fct = "tanh", rep = 1, threshold = 0.5
)
# create an converter for this model
converter <- Converter$new(nn)

# create new instance of 'LRP'
cw <- ConnectionWeights$new(converter)

library(plotly)

# Get the ggplot and add your changes
p <- plot(cw, output_idx = 1) +
  theme_bw() +
  scale_fill_gradient2(low = "green", mid = "black", high = "blue")

# Now apply the method plotly::ggplotly with argument tooltip = "text"
plotly::ggplotly(p, tooltip = "text")
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item J. D. Olden et al. (2004) \emph{An accurate comparison of methods for
quantifying variable importance in artificial neural networks using
simulated data.} Ecological Modelling 178, p. 389–397
}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{converter}}{The converter of class \link{Converter} with the stored and
torch-converted model.}

\item{\code{channels_first}}{The data format of the result, i.e. channels on
last dimension (\code{FALSE}) or on the first dimension (\code{TRUE}). If the
data has no channels, use the default value \code{TRUE}.}

\item{\code{dtype}}{The type of the data and parameters (either \code{'float'}
for \link[torch:torch_dtype]{torch::torch_float} or \code{'double'} for \link[torch:torch_dtype]{torch::torch_double}).}

\item{\code{result}}{The methods result as a torch tensor of size
\emph{(dim_in, dim_out)} and with data type \code{dtype}.}

\item{\code{output_idx}}{This vector determines for which outputs the method
will be applied. By default (\code{NULL}), all outputs (but limited to the
first 10) are considered.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{ConnectionWeights$new()}}
\item \href{#method-get_result}{\code{ConnectionWeights$get_result()}}
\item \href{#method-plot}{\code{ConnectionWeights$plot()}}
\item \href{#method-clone}{\code{ConnectionWeights$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ConnectionWeights$new(
  converter,
  output_idx = NULL,
  channels_first = TRUE,
  dtype = "float"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{converter}}{The converter of class \link{Converter} with the stored and
torch-converted model.}

\item{\code{output_idx}}{This vector determines for which output indices the
method will be applied. By default (\code{NULL}), all outputs (but limited
to the first 10) are considered.}

\item{\code{channels_first}}{The data format of the result, i.e. channels on
last dimension (\code{FALSE}) or on the first dimension (\code{TRUE}). If the
data has no channels, use the default value \code{TRUE}.}

\item{\code{dtype}}{The data type for the calculations. Use
either \code{'float'} for \link[torch:torch_dtype]{torch::torch_float} or \code{'double'} for
\link[torch:torch_dtype]{torch::torch_double}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_result"></a>}}
\if{latex}{\out{\hypertarget{method-get_result}{}}}
\subsection{Method \code{get_result()}}{
This function returns the result of the \emph{Connection Weights} method
either as an array (\code{'array'}), a torch tensor (\code{'torch.tensor'} or
\code{'torch_tensor'}) of size (dim_in, dim_out) or as a data.frame
(\code{'data.frame'}).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ConnectionWeights$get_result(type = "array")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{The data type of the result. Use one of \code{'array'},
\code{'torch.tensor'}, \code{'torch_tensor'} or \code{'data.frame'}
(default: \code{'array'}).}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The result of this method for the given data in the chosen
type.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-plot"></a>}}
\if{latex}{\out{\hypertarget{method-plot}{}}}
\subsection{Method \code{plot()}}{
This method visualizes the result of the \emph{ConnectionWeights} method in a
\link[ggplot2:ggplot]{ggplot2::ggplot}. You can use the argument \code{output_idx} to select
individual output nodes for the plot. The different results for the
selected outputs are visualized using the method \link[ggplot2:facet_grid]{ggplot2::facet_grid}.
You can also use the \code{as_plotly} argument to generate an interactive
plot based on the plot function \link[plotly:plot_ly]{plotly::plot_ly}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ConnectionWeights$plot(
  output_idx = NULL,
  aggr_channels = "sum",
  preprocess_FUN = identity,
  as_plotly = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{output_idx}}{An integer vector containing the numbers of the
output indices whose result is to be plotted, e.g. \code{c(1,4)} for the
first and fourth model output. But this vector must be included in the
vector \code{output_idx} from the initialization, otherwise, no results were
calculated for this output node and can not be plotted. By default
(\code{NULL}), the smallest index of all calculated output nodes is used.}

\item{\code{aggr_channels}}{Pass one of \code{'norm'}, \code{'sum'}, \code{'mean'} or a
custom function to aggregate the channels, e.g. the maximum
(\link[base:Extremes]{base::max}) or minimum (\link[base:Extremes]{base::min}) over the channels or only
individual channels with \code{function(x) x[1]}. By default (\code{'sum'}),
the sum of all channels is used.\cr
\strong{Note:} This argument is used only for 2D and 3D inputs.}

\item{\code{preprocess_FUN}}{This function is applied to the method's result
before generating the plot. By default, the identity function
(\code{identity}) is used.}

\item{\code{as_plotly}}{This boolean value (default: \code{FALSE}) can be used to
create an interactive plot based on the library \code{plotly}. This function
takes use of \link[plotly:ggplotly]{plotly::ggplotly}, hence make sure that the suggested
package \code{plotly} is installed in your R session.\cr
\strong{Advanced:} You can first
output the results as a ggplot (\code{as_plotly = FALSE}) and then make
custom changes to the plot, e.g. other theme or other fill color. Then
you can manually call the function \code{ggplotly} to get an interactive
plotly plot.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Returns either a \link[ggplot2:ggplot]{ggplot2::ggplot} (\code{as_plotly = FALSE}) or a
\link[plotly:plot_ly]{plotly::plot_ly} object (\code{as_plotly = TRUE}) with the plotted results.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ConnectionWeights$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
