#' Output the result of transition level intensity analysis as csv.
#' @details Gets the output of \code{TIA} function and the path variable and generate a csv report called "TransitionLevelIntensityAnalysis.csv". The output will be stored in "CSVOutput" directory in the path direction.
#' @param TIA.output Output list generated by \code{TIA} function.
#' @param filename A charachter variable including an optional path and a required filename to where the user wants to store the csv output. If only the name of the file is provided, it will be stored in the working directory.
#' @param time.points a charachter vector showing the time point of each raster layer in chronological order.
#' @param categories A charachter vector showing the categories in the map. Order of categories decided bases on the equivalent IDs in the raster attribute table.
#' @return The output is a CSV file.
#' @importFrom stats na.omit
#' @importFrom utils write.table
#' @export
#' @examples
#' raster_2005 <- raster::raster(system.file("external/RASTER_2005.RST", package="intensity.analysis"))
#' raster_2010 <- raster::raster(system.file("external/RASTER_2010.RST", package="intensity.analysis"))
#' raster_2012 <- raster::raster(system.file("external/RASTER_2012.RST", package="intensity.analysis"))
#' raster.layers <- list(raster_2005, raster_2010, raster_2012)
#' time.points <- c("2005","2010","2012")
#' categories <- c("Water","Trees","Impervious")
#' crosstabulation <- multicrosstab(raster.layers, time.points, categories)
#' TIA.output <- TIA(crosstabulation, time.points, categories)
#' filename <- file.path(normalizePath(tempdir(), winslash = "/"), "TIA.csv")
#' TIA2csv(TIA.output,time.points,categories, filename)


TIA2csv <- function(TIA.output,time.points,categories, filename){

  parameters <- reqpar(time.points)

  output <- chkfilename(filename, expand = TRUE)

  if (file.exists(output)){
    suppressWarnings(file.remove(output, showWarnings = FALSE ))
  }

  for (i in 1: as.integer(parameters$number.of.intervals)){
    transition.level.header <- paste("Transition level Intensity Analysis for interval: ",
                                     as.character(parameters$initial.times[i]),as.character("-"),as.character(parameters$subsequent.times[i]),sep =" ")
    cat(transition.level.header, "\n", file = output ,append = TRUE)

    for (x in 1: ncol(TIA.output$Annual.Transition.Intensity[[i]])){
      out.dataframe.2 <- cbind(na.omit(TIA.output$Annual.Transition.Size[[i]][ ,x]),na.omit(TIA.output$Annual.Transition.Intensity[[i]][ ,x]),
                               rep(TIA.output$Uniform.Transition[[i]][x],each= length(categories)-1),na.omit(TIA.output$Transition.Behavior.for.Gain[[i]][ ,x]))
      header5 <- categories[-x]
      rownames(out.dataframe.2) <- paste(header5)
      colnames(out.dataframe.2) <- c(paste("Annual Transition Size for Gain of", categories[x]), paste("Transition Intensity for Gain of ",categories[x]),
                                     "Uniform Transition Intensity",paste("Transition Behavior for Gain of ",categories[x]))
      # Add a header showing the interval for each dataframe
      # Write the transition level intensity analysis to a csv file
      write.table(out.dataframe.2, file = output, append = TRUE, col.names = NA ,quote = TRUE, sep = ",")
      cat("\n", file = output, append = TRUE)
    }
  }

}
