% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lp.R
\name{bound}
\alias{bound}
\title{Obtaining TE bounds}
\usage{
bound(env, sset, lpsolver, lpsolver.options, noisy = FALSE,
  smallreturnlist = FALSE, debug = FALSE)
}
\arguments{
\item{env}{environment containing the matrices defining the LP
problem.}

\item{sset}{a list containing the point estimates and gamma
components associated with each element in the S-set. This
object is only used to determine the names of terms. If it is
no submitted, then no names are provided to the solution
vector.}

\item{lpsolver}{string, name of the package used to solve the LP
problem.}

\item{lpsolver.options}{list, each item of the list should
correspond to an option specific to the LP solver selected.}

\item{noisy}{boolean, set to \code{TRUE} if optimization results
should be displayed.}

\item{smallreturnlist}{boolean, set to \code{TRUE} if the LP model
should not be returned.}

\item{debug}{boolean, indicates whether or not the function should
provide output when obtaining bounds. The option is only
applied when \code{lpsolver = 'gurobi'}. The output provided is
the same as what the Gurobi API would send to the console.}
}
\value{
a list containing the bounds on the treatment effect; the
    coefficients on each term in the MTR associated with the upper
    and lower bounds, for both counterfactuals; the optimization
    status to the maximization and minimization problems; the LP
    problem that the optimizer solved.
}
\description{
This function estimates the bounds on the target treatment
effect. The LP model must be passed as an environment variable,
under the entry \code{$lpobj}. See \code{\link{lpSetup}}.
}
\examples{
dtm <- ivmte:::gendistMosquito()

## Declare empty list to be updated (in the event multiple IV like
## specifications are provided
sSet <- list()

## Declare MTR formulas
formula0 = ~ 1 + u
formula1 = ~ 1 + u

## Construct object that separates out non-spline components of MTR
## formulas from the spline components. The MTR functions are
## obtained from this object by the function 'genSSet'.
splinesList = list(removeSplines(formula0), removeSplines(formula1))

## Construct MTR polynomials
polynomials0 <- polyparse(formula = formula0,
                          data = dtm,
                          uname = u,
                          as.function = FALSE)
polynomials1 <- polyparse(formula = formula1,
                          data = dtm,
                          uname = u,
                           as.function = FALSE)

## Generate propensity score model
propensityObj <- propensity(formula = d ~ z,
                            data = dtm,
                            link = "linear")

## Generate IV estimates
ivEstimates <- ivEstimate(formula = ey ~ d | z,
                          data = dtm,
                          components = l(intercept, d),
                          treat = d,
                          list = FALSE)

## Generate target gamma moments
targetGamma <- genTarget(treat = "d",
                         m0 = ~ 1 + u,
                         m1 = ~ 1 + u,
                         target = "atu",
                         data = dtm,
                         splinesobj = splinesList,
                         pmodobj = propensityObj,
                         pm0 = polynomials0,
                         pm1 = polynomials1,
                         point = FALSE)

## Construct S-set. which contains the coefficients and weights
## corresponding to various IV-like estimands
sSet <- genSSet(data = dtm,
                sset = sSet,
                sest = ivEstimates,
                splinesobj = splinesList,
                pmodobj = propensityObj$phat,
                pm0 = polynomials0,
                pm1 = polynomials1,
                ncomponents = 2,
                scount = 1,
                yvar = "ey",
                dvar = "d",
                means = TRUE)
## Only the entry $sset is required
sSet <- sSet$sset

## Define additional upper- and lower-bound constraints for the LP
## problem
A <- matrix(0, nrow = 22, ncol = 4)
A <- cbind(A, rbind(cbind(1, seq(0, 1, 0.1)),
                    matrix(0, nrow = 11, ncol = 2)))
A <- cbind(A, rbind(matrix(0, nrow = 11, ncol = 2),
                    cbind(1, seq(0, 1, 0.1))))
sense <- c(rep(">", 11), rep("<", 11))
rhs <- c(rep(0.2, 11), rep(0.8, 11))

## Construct LP object to be interpreted and solved by
## lpSolveAPI. Note that an environment has to be created for the LP
## object. The matrices defining the shape restrictions must be stored
## as a list under the entry \\code{$mbobj} in the environment.
lpEnv <- new.env()
lpEnv$mbobj <- list(mbA = A,
                    mbs = sense,
                    mbrhs = rhs)
## Convert the matrices defining the shape constraints into a format
## that is suitable for the LP solver.
lpSetup(env = lpEnv,
        sset = sSet,
        lpsolver = "lpsolveapi")
## Setup LP model so that it is solving for the bounds.
lpSetupBound(env = lpEnv,
             g0 = targetGamma$gstar0,
             g1 = targetGamma$gstar1,
             sset = sSet,
             criterion.factor = 0,
             lpsolver = "lpsolveapi")
## Declare any LP solver options as a list.
lpOptions <- optionsLpSolveAPI(list(epslevel = "tight"))
## Obtain the bounds.
bounds <- bound(env = lpEnv,
                sset = sSet,
                lpsolver = "lpsolveapi",
                lpsolver.options = lpOptions)
cat("The bounds are [",  bounds$min, ",", bounds$max, "].\\n")

}
