% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/varma.R
\name{estim.varma}
\alias{estim.varma}
\title{Estimate a VARMA Model}
\usage{
estim.varma(
  data,
  params = NULL,
  seasonsCount = 0,
  lbfgsOptions = get.options.lbfgs(),
  olsStdMultiplier = 2,
  pcaOptionsY = NULL,
  pcaOptionsX = NULL,
  maxHorizon = 0,
  simFixSize = 0,
  simHorizons = NULL,
  simUsePreviousEstim = FALSE,
  simMaxConditionNumber = Inf
)
}
\arguments{
\item{data}{A list that determines data and other required information for the search process.
Use \code{\link[=get.data]{get.data()}} function to generate it from a \code{matrix} or a \code{data.frame}.}

\item{params}{An integer vector that determines the values for the parameters of the VARMA model: \code{(p,d,q,P,D,Q)}. If \code{NULL}, \code{c(1,0,0,0,0,0)} is used.}

\item{seasonsCount}{An integer value representing the number of observations per unit of time.}

\item{lbfgsOptions}{A list containing L-BFGS optimization options. Use \link{get.options.lbfgs} function for initialization.}

\item{olsStdMultiplier}{A number used as a multiplier for the standard deviation of OLS, used for restricting maximum likelihood estimation.}

\item{pcaOptionsY}{A list of options to use principal components of \code{y}, instead of the actual values. Set to \code{NULL} to disable. Use \code{\link[=get.options.pca]{get.options.pca()}} for initialization.}

\item{pcaOptionsX}{A list of options to use principal components of \code{x}, instead of the actual values. Set to \code{NULL} to disable. Use \code{\link[=get.options.pca]{get.options.pca()}} for initialization.}

\item{maxHorizon}{An integer representing the maximum prediction horizon. Set to zero to disable prediction.}

\item{simFixSize}{An integer that determines the number of out-of-sample simulations. Use zero to disable simulation.}

\item{simHorizons}{An integer vector representing the prediction horizons to be used in out-of-sample simulations. See also \code{\link[=get.search.metrics]{get.search.metrics()}}.}

\item{simUsePreviousEstim}{If \code{TRUE}, parameters are initialized only in the first step of the simulation. The initial values of the n-th simulation (with one more observation) are the estimations from the previous step.}

\item{simMaxConditionNumber}{A number representing the maximum value for the condition number in simulation.}
}
\value{
A nested list with the following items:
\item{counts}{Information about different aspects of the estimation such as the number of observation, number of exogenous variables, etc.}
\item{estimations}{Estimated coefficients, standard errors, z-statistics, p-values, etc.}
\item{metrics}{Value of different goodness of fit and out-of-sample performance metrics. }
\item{prediction}{Information on the predicted values.}
\item{simulation}{Information on the simulations. }
\item{info}{Some other general information.}
}
\description{
Use this function to estimate a Vector Autoregressive Moving Average model.
}
\details{
The VARMA model can be used to analyze multivariate time series data with seasonal or non-seasonal patterns. According to \insertCite{lutkepohl2005new;textual}{ldt}, it considers interdependencies between the series, making it a powerful tool for prediction. The specification of this model is given by:
\deqn{
\Delta^d \Delta_s^D y_t = c + \sum_{i=1}^p A_i y_{t-i} + \sum_{i=1}^q B_i \epsilon_{t-i} + C x_t + \sum_{i=1}^P A_{is} y_{t-is} + \sum_{i=1}^Q B_{is} v_{t-is} + v_t,
}
where \eqn{y_t:m\times 1} is the vector of endogenous variables, \eqn{x_t:k\times 1} is the vector exogenous variables, \eqn{s} is the number of seasons and \eqn{(p,d,q,P,D,Q)} determines the lag structure of the model. Furthermore, \eqn{c,C,A_i} and \eqn{B_i} for all available \eqn{i} determines the model’s parameters. \eqn{v_t} is the disturbance vector and is contemporaneously correlated between different equations, i.e., \eqn{E(v_tv_t')=\Sigma}.
Given a sample of size \eqn{T}, the model can be estimated using maximum likelihood estimation. However, to ensure identifiability, it is necessary to impose additional constraints on the parameters (see chapter 12 in \insertCite{lutkepohl2005new;textual}{ldt}). In this function, diagonal MA equation form is used (see \insertCite{dufour2022practical;textual}{ldt}).
In this function, the feasible GLS estimator is used to initialize the maximum likelihood, and the OLS estimator is used to calculate the initial value of the variance matrix of the error term. The condition number is calculated similar to the other models (see \link{estim.sur} or e.g., page 94 in \insertCite{trefethen1997numerical;textual}{ldt}). Furthermore, given a prediction horizon and required exogenous data, prediction is performed in a recursive schema, in which the actual estimated errors are used if available and zero otherwise. The variance of the predictions is also calculated recursively. Note that this function does not incorporate the coefficients uncertainty in calculation of the variance (see section 3.5 in \insertCite{lutkepohl2005new;textual}{ldt}).

Finally, note that the main purpose of exporting this method is to show the inner calculations of the search process in \link{search.varma} function.
}
\examples{
# Example 1 (simulation, ARMA):
num_eq <- 1L
num_ar <- 2L
num_ma <- 1L
num_exo <- 1L
sample <- sim.varma(num_eq, arList = num_ar, maList = num_ma, exoCoef = num_exo, nObs = 110)
# estimate:
fit <- estim.varma(data = get.data(cbind(sample$y, sample$x)[1:100,],
                                   endogenous = num_eq,
                                   newData = sample$x[101:110,, drop=FALSE]),
                   params = c(num_ar, 0, num_ma, 0, 0, 0),
                   maxHorizon = 10,
                   simFixSize = 5,
                   simHorizons = c(1:10))
print(fit)

pred <- predict(fit, actualCount = 10)
plot(pred, simMetric = "mape")

# split coefficient matrix:
get.varma.params(fit$estimations$coefs, numAR = num_ar, numMA = num_ma, numExo = num_exo)

# Example 2 (simulation, VARMA):
num_eq <- 3L
num_ar <- 2L
num_ma <- 1L
num_ma <- 1L
num_exo <- 2L
sample <- sim.varma(num_eq, arList = num_ar, maList = num_ma, exoCoef = num_exo, nObs = 110)
# estimate:
fit <- estim.varma(data = get.data(cbind(sample$y, sample$x)[1:100,],
                                   endogenous = num_eq,
                                   newData = sample$x[101:110,]),
                   params = c(num_ar, 0, num_ma, 0, 0, 0),
                   maxHorizon = 10,
                   simFixSize = 5,
                   simHorizons = c(1:10))

pred <- predict(fit, actualCount = 10)
plot(pred, simMetric = "mape")

# split coefficient matrix:
get.varma.params(fit$estimations$coefs, numAR = num_ar, numMA = num_ma, numExo = num_exo)
}
\references{
\insertAllCited{}
}
\seealso{
\link{search.varma}
}
