% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/demeanlist.R
\name{demeanlist}
\alias{demeanlist}
\title{Centre vectors on multiple groups}
\usage{
demeanlist(
  mtx,
  fl,
  icpt = 0L,
  eps = getOption("lfe.eps"),
  threads = getOption("lfe.threads"),
  progress = getOption("lfe.pint"),
  accel = getOption("lfe.accel"),
  randfact = TRUE,
  means = FALSE,
  weights = NULL,
  scale = TRUE,
  na.rm = FALSE,
  attrs = NULL
)
}
\arguments{
\item{mtx}{matrix whose columns form vectors to be group-centred. mtx
can also be a list of vectors or matrices, such as a data frame.}

\item{fl}{list of factors defining the grouping structure.}

\item{icpt}{the position of the intercept, this column is removed from
the result matrix.}

\item{eps}{a tolerance for the centering.}

\item{threads}{an integer specifying the number of threads to use.}

\item{progress}{integer. If positive, make progress reports (whenever a
vector is centered, but not more often than every \code{progress} minutes).}

\item{accel}{integer. Set to 1 if Gearhart-Koshy acceleration should be done.}

\item{randfact}{logical. Should the order of the factors be randomized?
This may improve convergence.}

\item{means}{logical. Should the means instead of the demeaned matrix be
returned? Setting \code{means=TRUE} will return \code{mtx -  demeanlist(mtx,...)},
but without the extra copy.}

\item{weights}{numeric. For weighted demeaning.}

\item{scale}{logical. Specify scaling for weighted demeaning.}

\item{na.rm}{logical which indicates what should happen when the data
contain \code{NA}s. If TRUE, rows in the input \code{mtx} are removed
prior to centering. If FALSE, they are kept, leading to entire groups becoming NA
in the output.}

\item{attrs}{list. List of attributes which should be attached to the output.
Used internally.}
}
\value{
If \code{mtx} is a matrix, a matrix of the same shape, possibly with
column \code{icpt} deleted.

If \code{mtx} is a list of vectors and matrices, a list of the same
length is returned, with the same vector and matrix-pattern, but the
matrices have the column \code{icpt} deleted.

If \code{mtx} is a \code{'data.frame'}, a \code{'data.frame'}
with the same names is returned; the \code{icpt} argument is ignored.

If \code{na.rm} is specified, the return value has an attribute \code{'na.rm'} with a vector of
row numbers which has been removed. In case the input is a matrix or list, the same rows
are removed from all of them. Note that removing NAs incurs a copy of the input, so if
memory usage is an issue and many runs are done, one might consider removing NAs from the data set entirely.
}
\description{
Uses the method of alternating projections to centre
a (model) matrix on multiple groups, as specified by a list of factors.
This function is called by \code{\link[=felm]{felm()}}, but it has been
made available as standalone in case it's needed. In particular, if
one does not need transformations provided by R-formulas but have the covariates present
as a matrix or a data.frame, a substantial amount of time can be saved in the centering.
}
\details{
For each column \code{y} in \code{mtx}, the equivalent of the
following centering is performed, with \code{cy} as the result.
\preformatted{
cy <- y; oldy <- y-1
while(sqrt(sum((cy-oldy)**2)) >= eps) {
 oldy <- cy
 for(f in fl) cy <- cy - ave(cy,f)
}
}

Each factor in \code{fl} may contain an
attribute \code{'x'} which is a numeric vector of the same length as
the factor. The centering is then not done on the means of each group,
but on the projection onto the covariate in each group.  That is, with a
covariate \code{x} and a factor \code{f}, it is like projecting out the
interaction \code{x:f}.  The \code{'x'} attribute can also be a matrix of column
vectors, in this case it can be beneficial to orthogonalize the columns,
either with a stabilized Gram-Schmidt method, or with the simple
method \verb{x \\\%*\\\% solve(chol(crossprod(x)))}.

The \code{weights} argument is used if a weighted projection is
computed.  If \eqn{W} is the diagonal matrix with \code{weights} on the
diagonal, \code{demeanlist} computes \eqn{W^{-1} M_{WD} W x} where \eqn{x} is
the input vector, \eqn{D} is the matrix of dummies from \code{fl} and
\eqn{M_{WD}} is the projection on the orthogonal complement of
the range of the matrix \eqn{WD}.  It is possible to implement the
weighted projection with the \code{'x'} attribute mentioned above, but
it is a separate argument for convenience.
If \code{scale=FALSE}, \code{demeanlist} computes \eqn{M_{WD} x} without
any \eqn{W} scaling.
If \code{length(scale) > 1}, then \code{scale[1]} specifies whether
the input should be scaled by \eqn{W}, and \code{scale[2]} specifies
whether the output should be scaled by \eqn{W^{-1}}.  This is just
a convenience to save some memory copies in other functions in the package.

Note that for certain large datasets the overhead in \code{\link[=felm]{felm()}}
is large compared to the time spent in \code{demeanlist}. If the data
are present directly without having to use the formula-interface to
\code{felm} for transformations etc, it is possible to run
\code{demeanlist} directly on a matrix or \code{"data.frame"} and do the
OLS "manually", e.g. with something like
\verb{cx <- demeanlist(x,...); beta <- solve(crossprod(cx), crossprod(cx,y))}

In some applications it is known that a single centering iteration is
sufficient. In particular, if \code{length(fl)==1} and there is no
interaction attribute \code{x}.  In this case the centering algorithm is
terminated after the first iteration. There may be other cases, e.g. if
there is a single factor with a matrix \code{x} with orthogonal columns. If
you have such prior knowledge, it is possible to force termination after
the first iteration by adding an attribute \code{attr(fl, 'oneiter') <- TRUE}.  Convergence will be reached in the second iteration anyway, but
you save one iteration, i.e. you double the speed.
}
\note{
The \code{accel} argument enables Gearhart-Koshy acceleration as
described in Theorem 3.16 by Bauschke, Deutsch, Hundal and Park in "Accelerating the
convergence of the method of alternating projections",
Trans. Amer. Math. Soc. 355 pp 3433-3461 (2003).

\code{demeanlist} will use an in place transform to save memory, provided the \code{mtx}
argument is unnamed. Thus, as always in R, you shouldn't use temporary variables
like \verb{tmp <- fun(x[v,]); bar <- demeanlist(tmp,...); rm(tmp)}, it will be much better to
do \code{bar <- demeanlist(fun(x[v,]),...)}. However, demeanlist allows a construction like
\code{bar <- demeanlist(unnamed(tmp),...)} which will use an in place transformation, i.e. tmp
will be modified, quite contrary to the usual semantics of R.
}
\examples{
oldopts <- options("lfe.threads")
options(lfe.threads = 2)
## create a matrix
mtx <- data.frame(matrix(rnorm(999), ncol = 3))
# a list of factors
rgb <- c("red", "green", "blue")
fl <- replicate(4, factor(sample(rgb, nrow(mtx), replace = TRUE)), simplify = FALSE)
names(fl) <- paste("g", seq_along(fl), sep = "")
# centre on all means
mtx0 <- demeanlist(mtx, fl)
head(data.frame(mtx0, fl))
# verify that the group means for the columns are zero
lapply(fl, function(f) apply(mtx0, 2, tapply, f, mean))
options(oldopts)
}
