% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spa.R
\name{project}
\alias{project}
\title{Project a vector on the border of the region defined by a set of linear (in)equality restrictions.}
\usage{
project(x, A, b, neq = length(b), w = rep(1, length(x)), eps = 0.01,
  maxiter = 1000L)
}
\arguments{
\item{x}{[\code{numeric}] Vector that needs to satisfy the linear restrictions.}

\item{A}{[\code{matrix}] Coefficient matrix for linear restrictions.}

\item{b}{[\code{numeric}] Right hand side of linear restrictions.}

\item{neq}{[\code{numeric}] The first \code{neq} rows in \code{A} and \code{b} are treated as linear equalities. 
The others as Linear inequalities of the form \eqn{Ax<=b}.}

\item{w}{[\code{numeric}] Optional weight vector of the same length as \code{x}. Must be positive.}

\item{eps}{The maximum allowed deviation from the constraints (see details).}

\item{maxiter}{maximum number of iterations}
}
\value{
A \code{list} with the following entries:
\itemize{
 \item{\code{x}: the adjusted vector}
 \item{\code{status}: Exit status:
  \itemize{
   \item{0: success}
   \item{1: could not allocate enough memory (space for approximately \eqn{2(m+n)} \code{double}s is necessary).}
   \item{2: divergence detected (set of restrictions may be contradictory)}
   \item{3: maximum number of iterations reached}
  }
 }
 \item{\code{eps}: The tolerance achieved after optimizing (see Details).}
 \item{\code{iterations}: The number of iterations performed.}
 \item{\code{duration}: the time it took to compute the adjusted vector}
 \item{\code{objective}: The (weighted) Euclidean distance between the initial and the adjusted vector}
}
}
\description{
Compute a vector, closest to \code{x} in the Euclidean sense, satisfying a set of linear (in)equality restrictions.
}
\section{Details}{


The tolerance \code{eps} is defined as the maximum absolute value of the difference vector 
\eqn{\boldsymbol{Ax}-\boldsymbol{b}} for equalities. For inequalities, the difference vector
is set to zero when it's value is lesser than zero (i.e. when the restriction is satisfied). The
algorithm iterates until either the tolerance is met, the number of allowed iterations is
exceeded or divergence is detected.
}

\examples{

# the system 
# x + y = 10
# -x <= 0   # ==> x > 0
# -y <= 0   # ==> y > 0
#
A <- matrix(c(
   1,1,
  -1,0,
   0,-1), byrow=TRUE, nrow=3
)
b <- c(10,0,0)

# x and y will be adjusted by the same amount
project(x=c(4,5), A=A, b=b, neq=1)

# One of the inequalies violated
project(x=c(-1,5), A=A, b=b, neq=1)

# Weighted distances: 'heavy' variables change less
project(x=c(4,5), A=A, b=b, neq=1, w=c(100,1))

# if w=1/x0, the ratio between coefficients of x0 stay the same (to first order)
x0 <- c(x=4,y=5)
x1 <- project(x=x0,A=A,b=b,neq=1,w=1/x0)

x0[1]/x0[2]
x1$x[1] / x1$x[2]



}
\seealso{
\code{\link{sparse_project}}
}
