% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lmw_iv.R
\name{lmw_iv}
\alias{lmw_iv}
\title{Compute instrumental variable regression-implied weights}
\usage{
lmw_iv(
  formula,
  data = NULL,
  estimand = "ATE",
  method = "URI",
  treat = NULL,
  iv,
  base.weights = NULL,
  s.weights = NULL,
  obj = NULL,
  fixef = NULL,
  target = NULL,
  target.weights = NULL,
  contrast = NULL,
  focal = NULL
)
}
\arguments{
\item{formula}{a one-sided formula with the treatment and covariates on the
right-hand side corresponding to the second-stage (reduced form) outcome
regression model to be fit. If an outcome variable is supplied on the
left-hand side, it will be ignored. This model should not include an IV. See
Details for how this formula is interpreted in light of other options.}

\item{data}{a data frame containing the variables named in \code{formula},
\code{treat}, and \code{iv}.}

\item{estimand}{the estimand of interest, which determines how covariates
are centered. Should be one of \code{"ATE"} for the average treatment
effect, \code{"ATT"} for the average treatment effect in the treated,
\code{"ATC"} for the average treatment effect in the control, or
\code{"CATE"} for the conditional average treatment effect. When
\code{estimand = "CATE"}, an argument to \code{target} must be supplied.
This argument also affects what \code{\link[=summary.lmw]{summary.lmw()}} considers to be
the target population. Default is \code{"ATE"} unless \code{obj} is
specified, in which case it takes its value from the supplied object.}

\item{method}{the method used to estimate the weights; either \code{"URI"}
(the default) for uni-regression imputation weights, where a single model is
fit to the whole dataset, or \code{"MRI"} for multi-regression imputation,
where the covariates fully interact with the treatment. This affects the
interpretation of \code{formula}. See Details.}

\item{treat}{the name of the treatment variable in \code{data}. If
unspecified, the first variable present in \code{formula} will be taken as
the treatment variable with a message. Currently, only binary treatments are
supported. See Details.}

\item{iv}{a character vector or one-sided formula containing the names of
the IVs in \code{data}. These variables should not appear in \code{formula}.
Multiple IVs are allowed. See Details. This argument is required.}

\item{base.weights}{a vector of base weights. See Details. If omitted and
\code{obj} is specified, the weights from the supplied object will be used.
Can be supplied as a numeric vector, a string containing the name of the
variable in \code{data} containing the base weights, or the unquoted name of
the variable in \code{data} containing the base weights.}

\item{s.weights}{a vector of sampling weights. See Details. If omitted and
\code{obj} is specified, the sampling weights from the supplied object will
be used. Can be supplied as a numeric vector, a string containing the name
of the variable in \code{data} containing the sampling weights, or the
unquoted name of the variable in \code{data} containing the sampling
weights.}

\item{obj}{a \code{matchit} or \code{weightit} object corresponding to the
matched or weighted sample in which the implied IV regression would take
place. See Details.}

\item{fixef}{optional; a string or one-sided formula containing the name of
the fixed effects variable in \code{data}. See Details.}

\item{target}{a list or data frame containing the target values for each
covariate included in \code{formula}. Ignored with a warning when
\code{estimand} is not \code{"CATE"}.}

\item{target.weights}{a vector of sampling weights to be applied to
\code{target} when supplied as a data frame. Ignored with a warning when
\code{estimand} is not \code{"CATE"}.}

\item{contrast}{ignored.}

\item{focal}{the level of the treatment variable to be considered "focal"
(i.e., the "treated" level when \code{estimand = "ATT"} or the control level
when \code{estimand = "ATC"}). Ignored when \code{estimand} is \code{"ATE"}
or \code{"CATE"}. For binary treatments, this generally does not need to be
supplied.}
}
\value{
An \code{lmw_iv} object, which inherits from \code{lmw} objects and
contains the following components:
\item{treat}{the treatment variable,
given as a factor.}
\item{weights}{the computed implied regression weights.}
\item{covs}{a data frame containing the covariates included the model
formula.}
\item{estimand}{the requested estimand.}
\item{method}{the method
used to estimate the weights (\code{"URI"} or \code{"MRI"}).}
\item{base.weights}{the weights supplied to \code{base.weights}.}
\item{s.weights}{the weights supplied to \code{s.weights}.}
\item{call}{the
original call to \code{lmw_iv()}.}
\item{fixef}{the fixed effects variable
if supplied to \code{fixef}.}
\item{formula}{the model formula.}
\item{iv}{the instrumental variables, given as a one-sided formula.}
\item{target}{the supplied covariate target values when \code{estimand = "CATE"}, after some initial processing.}
\item{contrast}{the contrasted
treatment groups.}
\item{focal}{the focal treatment levels when
\code{estimand} is \code{"ATT"} or \code{"ATC"}.}

All functions that lack a specific \code{lmw_iv} method work with
\code{lmw_iv} objects as they do for \code{lmw} objects, such as
\code{\link[=summary.lmw]{summary.lmw()}}, \code{\link[=plot.lmw]{plot.lmw()}}, etc.
}
\description{
Computes the weights implied by an instrumental variable (IV) model that
would estimate a weighted difference in outcome means equal to the treatment
effect resulting from the supplied model fit with two-stage least squares.
}
\details{
\code{lmw_iv()} computes weights that make the weighted difference in
outcome means between the treatment groups equal to the two-stage least
squares (2SLS) estimate of the treatment effect. \code{formula} corresponds
to the second-stage (reduced form) model, with the treatment replaced by its
fitted values resulting from the first stage model. The first stage is fit
by replacing the treatment in the supplied \code{formula} with the IVs named
in \code{iv} and using the treatment as the outcome. The treatment is
assumed to be endogenous and the supplied instrumental variables assumed to
be instruments conditional on the other covariates, which are assumed to to
be exogenous.

When any treatment-by-covariate interactions are present in \code{formula}
or when \code{method = "MRI"}, covariates are centered at specific values to
ensure the resulting weights correspond to the desired estimand as supplied
to the \code{estimand} argument. For the ATE, the covariates are centered at
their means in the full sample. For the ATT and ATC, the covariates are
centered at their means in the treatment or control group (i.e., the
\code{focal} group), respectively. For the CATE, the covariates are centered
according to the argument supplied to \code{target} (see below). Note that
when covariate-by-covariate interactions are present, they will be centered
after computing the interaction rather than the interaction being computed
on the centered covariates unless \code{estimand = "CATE"}, in which case
the covariates will be centered at the values specified in \code{target}
prior to involvement in interactions. Note that the resulting effect estimate
does not actually correspond to the estimand supplied unless all effect
heterogeneity is due to the included covariates.

When treatment-by-covariate interactions are included in \code{formula},
additional instruments will be formed as the product of the supplied IVs and
the interacting covariates. When \code{method = "MRI"}, instruments will be
formed as the product of the supplied IVs and each of the covariates. All
treatment-by-covariate interactions are considered endogenous.

\subsection{Base weights and sampling weights}{

Base weights (\code{base.weights}) and sampling weights (\code{s.weights})
are similar in that they both involve combining weights with an outcome
regression model. However, they differ in a few ways. Sampling weights are
primarily used to adjust the target population; when the outcome model is
fit, it is fit using weighted least squares, and when target balance is
assessed, it is assessed using the sampling weighted population as the
target population. Centering of covariates in the outcome model is done
using the sampling weighted covariate means. Base weights are primarily used
to offer a second level of balancing beyond the implied regression weights,
i.e., to fit the 2SLS models in the base-weighted sample. Base weights do
not change the target population, so when target balance is assessed, it is
assessed using the unweighted population as the target population.

Some forms of weights both change the target population and provide an extra
layer of balancing, like propensity score weights that target estimands
other than the ATT, ATC, or ATE (e.g., overlap weights), or matching weights
where the target population is defined by the matching (e.g., matching with
a caliper, cardinality matching, or coarsened exact matching). Because these
weights change the target population, they should be supplied to
\code{s.weights} to ensure covariates are appropriately centered. In
\code{lmw_iv()}, whether weights are supplied to \code{base.weights} or
\code{s.weights} will not matter for the estimation of the weights but will
affect the target population in \link[=summary.lmw]{balance assessment}.

When both \code{base.weights} and \code{s.weights} are supplied, e.g., when
the base weights are the result of a propensity score model fit with
sampling weights, it is assumed the base weights do not incorporate the
sampling weights; that is, it is assumed that to estimate a treatment effect
\emph{without} regression adjustment, the base weights and the sampling
weights would have to be multiplied together. This is true, for example, for
the weights in a \code{matchit} or \code{weightit} object (see below) but
not for weights in the output of \code{MatchIt::match.data()} unless called
with \code{include.s.weights = FALSE} or weights resulting from
\code{CBPS::CBPS()}.
}

\subsection{2SLS after using \pkg{MatchIt} or \pkg{WeightIt}}{
Instrumental variable regression weights can be computed in a matched or weighted sample
by supplying a \code{matchit} or \code{weightit} object (from \pkg{MatchIt}
or \pkg{WeightIt}, respectively) to the \code{obj} argument of \code{lmw()}.
The estimand, base weights, and sampling weights (if any) will be taken from
the supplied object and used in the calculation of the implied regression
weights, unless these have been supplied separately to \code{lmw_iv()}. The
\code{weights} component of the supplied object containing the matching or
balancing weights will be passed to \code{base.weights} and the
\code{s.weights} component will be passed to \code{s.weights}. Arguments
supplied to \code{lmw_iv()} will take precedence over the corresponding
components in the \code{obj} object.
}

\subsection{Multi-category treatments}{
Multi-category treatments are not
currently supported for \code{lmw_iv()}.
}

\subsection{Fixed effects}{
A fixed effects variable can be supplied to the
\code{fixef} argument. This is equivalent to adding the fixed effects
variable as an exogenous predictor that does not interact with the
treatment, IV, or any other covariate. The difference is that computation is
much faster when the fixed effect has many levels because demeaning is used
rather than including the fixed effect variable as a collection of dummy
variables. When using URI, the weights will be the same regardless of
whether the fixed effect variable is included as a covariate or supplied to
\code{fixef}; when using MRI, results will differ because the fixed effect
variable does not interact with treatment. The fixed effects variable will
not appear in the \code{\link[=summary.lmw]{summary.lmw()}} output (but can be added using
\code{addlvariables} argument) or in the model output of \code{\link[=lmw_est]{lmw_est()}} or
\code{\link[=summary.lmw_est]{summary.lmw_est()}}. Because it does not interact with the
treatment, the distribution of the fixed effect variable may not correspond
to the target population, so caution should be used if it is expected the
treatment effect varies across levels of this variable (in which case it
should be included as a predictor). Currently only one fixed effect variable
is allowed.
}
}
\examples{
# URI for the ATT using instrument `Ins`
lmw.out <- lmw_iv(~ treat + age + education + race +
                    re74, data = lalonde,
                  estimand = "ATT", method = "URI",
                  treat = "treat", iv = ~Ins)
lmw.out
summary(lmw.out, addlvariables = ~married + re75)
}
\references{
Chattopadhyay, A., & Zubizarreta, J. R. (2022). On the implied
weights of linear regression for causal inference. \emph{Biometrika}, asac058. \doi{10.1093/biomet/asac058}
}
\seealso{
\code{\link[=summary.lmw]{summary.lmw()}} for summarizing balance and
representativeness; \code{\link[=plot.lmw]{plot.lmw()}} for plotting features of the
weights; \code{\link[=lmw_est]{lmw_est()}} for estimating treatment effects from
\code{lmw_iv} objects; \code{\link[=influence.lmw]{influence.lmw()}} for influence measures;
\code{ivreg()} in the \pkg{ivreg} package for fitting 2SLS models.
}
