% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lp_nl.R
\name{lp_nl}
\alias{lp_nl}
\title{Compute nonlinear impulse responses}
\usage{
lp_nl(data_set_df, lags_lin = NULL, lags_nl = NULL, lags_criterion = NULL,
  max_lags = NULL, trend = NULL, shock_type = NULL, confint = NULL,
  hor = NULL, switching = NULL, hp_filter = NULL, lambda = NULL,
  gamma = NULL)
}
\arguments{
\item{data_set_df}{A \link{data.frame}() containing all endogenous variables for the VAR. The column order
is used for the Cholesky decomposition.}

\item{lags_lin}{NaN or integer. NaN if lag length criterion is used.
Integer for number of lags for linear VAR to identify shock.}

\item{lags_nl}{NaN or integer. Number of lags for nonlinear VAR (if \emph{lags_criterion} = NaN). NaN if lag length criterion is given.}

\item{lags_criterion}{NaN or character. NaN means that the number of lags
will be given at \emph{lags_nl} and \emph{lags_lin}. The lag length criteria are 'AICc', 'AIC' and 'BIC'.}

\item{max_lags}{NaN or integer. Maximum number of lags (if \emph{lags_criterion} = 'AICc', 'AIC', 'BIC'). NaN otherwise.}

\item{trend}{Integer. Include no trend =  0 , include trend = 1, include trend and quadratic trend = 2.}

\item{shock_type}{Integer. Standard deviation shock = 0, unit shock = 1.}

\item{confint}{Double. Width of confidence bands. 68\% = 1; 90\% = 1.65; 95\% = 1.96.}

\item{hor}{Integer. Number of horizons for impulse responses.}

\item{switching}{Numeric vector. A column vector with the same length as \emph{data_set_df}. This series can either
be decomposed via the Hodrick-Prescott filter (see Auerbach and Gorodnichenko, 2013) or
directly plugged into the following smooth transition function:
\deqn{ F_{z_t} = \frac{exp(-\gamma z_t)}{1 + exp(-\gamma z_t)} }
Warning: \eqn{F_{z_t}} will be lagged in \link{create_nl_data} by one and then multiplied with the data.
If the variable shall not be lagged, the vector has to be given with a lead of one.
The data for the two regimes are: \cr
Regime 1 = (1-\eqn{F(z_{t-1})})*y_{t-p}, \cr
Regime 2 = \eqn{F(z_{t-1})}*y_{t-p}.}

\item{hp_filter}{Integer. No HP-filter = 0. Use HP-filter = 1.}

\item{lambda}{Double. Value of \eqn{\lambda} for the Hodrick-Prescott filter if HP-filter is applied.}

\item{gamma}{Double. Positive number which is used in the transition function.}
}
\value{
A list with impulse responses and their robust confidence bands.
It also contains a list named \emph{specs} with properties of \emph{data_set_df} for the plot function.

\item{irf_s1_mean}{A three 3D \link{array}() containing all impulse responses for all endogenous variables of the first state.
                   The last dimension denotes the shock variable. The row in each matrix
                   denotes the responses of the \emph{ith} variable, ordered as in \emph{data_set_df}. The columns are the horizons.
                   For example, if the results are saved in \emph{results_nl}, results_nl$irf_s1_mean[, , 1] returns a KXH matrix,
                   where K is the number of variables and H the number of horizons. '1' is the shock variable, corresponding to the
                   variable in the first column of \emph{data_set_df}.}

\item{irf_s1_low}{A three 3D \link{array}() containing all lower confidence bands of the impulse responses, based on
                   robust standard errors by Newey and West (1987). Properties are equal to \emph{irf_s1_mean}.}

\item{irf_s1_up}{A three 3D \link{array}() containing all upper confidence bands of the impulse responses, based on
                   robust standard errors by Newey and West (1987). Properties are equal to \emph{irf_s1_mean}.}

\item{irf_s2_mean}{A three 3D \link{array}() containing all impulse responses for all endogenous variables of the second state.
                   The last dimension denotes the shock variable. The row in each matrix
                   denotes the responses of the \emph{ith} variable, ordered as in data_set_df. The columns denote the horizon.
                   For example, if the results are saved in \emph{results_nl}, results_nl$irf_s2_mean[, , 1] returns a KXH matrix,
                   where K is the number of variables and H the number of horizons. '1' is the first shock variable corresponding to the
                   variable in the first column of \emph{data_set_df}.}

\item{irf_s2_low}{A three 3D \link{array}() containing all lower confidence bands of the responses,
                   based on robust standard errors by Newey and West (1987). Properties are equal to \emph{irf_s2_mean}.}

\item{irf_s2_up}{A three 3D \link{array}(), containing all upper confidence bands of the responses, based on
                   robust standard errors by Newey and West (1987). Properties are equal to \emph{irf_s2_mean}.}

\item{specs}{A list with properties of \emph{data_set_df} for the plot function.}

\item{fz}{A vector containing the values of the transition function F(z_{t-1}).}
}
\description{
Compute nonlinear impulse responses with local projections by Jordà (2005). The
data are separated into two states via a smooth transition function, applied in Auerbach and Gorodnichenko (2012).
}
\examples{
\donttest{
# Load package
  library(lpirfs)

# Load data
  data_set_df <- monetary_var_data

# Estimate model and save results
  results_nl <- lp_nl(data_set_df, lags_lin       = 4L,
                                   lags_nl        = 3L,
                                   lags_criterion = NaN,
                                   max_lags       = NaN,
                                   trend          = 0L,
                                   shock_type     = 1L,
                                   confint        = 1.96,
                                   hor            = 24L,
                                   switching      = data_set_df$FF,
                                   hp_filter      = 1L,
                                   lambda         = 1600,
                                   gamma          = 3)

# Make and save all plots
  nl_plots <- plot_nl_irfs(results_nl)

# Show all plots
  library(ggpubr)
  library(gridExtra)

# Save plots based on states
  s1_plots <- sapply(nl_plots$gg_s1, ggplotGrob)
  s2_plots <- sapply(nl_plots$gg_s2, ggplotGrob)

# Show first irf of each state
  plot(s1_plots[[1]])
  plot(s2_plots[[1]])

# Show all plots
  marrangeGrob(s1_plots, nrow = ncol(data_set_df), ncol = ncol(data_set_df), top = NULL)
  marrangeGrob(s2_plots, nrow = ncol(data_set_df), ncol = ncol(data_set_df), top = NULL)

}
}
\references{
Akaike, H. (1974). "A new look at the statistical model identification", \emph{IEEE Transactions on Automatic Control}, 19 (6): 716–723.

Auerbach, A. J., and  Gorodnichenko Y. (2012). "Measuring the Output Responses to Fiscal Policy."
\emph{American Economic Journal: Economic Policy}, 4 (2): 1-27.

Auerbach, A. J., and Gorodnichenko Y. (2013). "Fiscal Multipliers in Recession and Expansion."
\emph{NBER Working Paper Series}. Nr 17447.

Hurvich, C. M., and Tsai, C.-L. (1993) “A Corrected Akaike Information Criterion for
Vector Autoregressive Model Selection.” \emph{Journal of Time Series Analysis}, 1993, 14(3):
271–79.

Jordà, Ò. (2005) "Estimation and Inference of Impulse Responses by Local Projections."
\emph{American Economic Review}, 95 (1): 161-182.

Newey W.K., and West K.D. (1987). “A Simple, Positive-Definite, Heteroskedasticity and
Autocorrelation Consistent Covariance Matrix.” \emph{Econometrica}, 55, 703–708.

Schwarz, Gideon E. (1978). "Estimating the dimension of a model", \emph{Annals of Statistics}, 6 (2): 461–464.
}
\author{
Philipp Adämmer
}
