% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/esp_get_capimun.R
\name{esp_get_capimun}
\alias{esp_get_capimun}
\title{Get \pkg{sf} points of the municipalities of Spain}
\source{
IGN data via a custom CDN (see
\url{https://github.com/rOpenSpain/mapSpain/tree/sianedata}).
}
\usage{
esp_get_capimun(
  year = Sys.Date(),
  epsg = "4258",
  cache = TRUE,
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  region = NULL,
  munic = NULL,
  moveCAN = TRUE,
  rawcols = FALSE
)
}
\arguments{
\item{year}{Release year. See \strong{Details} for years available.}

\item{epsg}{projection of the map: 4-digit \href{https://epsg.io/}{EPSG code}.
One of:
\itemize{
\item \code{"4258"}: ETRS89
\item \code{"4326"}: WGS84
\item \code{"3035"}: ETRS89 / ETRS-LAEA
\item \code{"3857"}: Pseudo-Mercator
}}

\item{cache}{A logical whether to do caching. Default is \code{TRUE}. See
\strong{About caching}.}

\item{update_cache}{A logical whether to update cache. Default is \code{FALSE}.
When set to \code{TRUE} it would force a fresh download of the source file.}

\item{cache_dir}{A path to a cache directory. See \strong{About caching}.}

\item{verbose}{Logical, displays information. Useful for debugging,
default is \code{FALSE}.}

\item{region}{A vector of names and/or codes for provinces
or \code{NULL} to get all the municipalities. See \strong{Details}.}

\item{munic}{A name or \code{\link[base:grep]{regex}} expression with the names of
the required municipalities. \code{NULL} would not produce any filtering.}

\item{moveCAN}{A logical \code{TRUE/FALSE} or a vector of coordinates
\code{c(lat, lon)}. It places the Canary Islands close to Spain's mainland.
Initial position can be adjusted using the vector of coordinates. See
\strong{Displacing the Canary Islands}.}

\item{rawcols}{Logical. Setting this to \code{TRUE} would add the raw columns of
the dataset provided by IGN.}
}
\value{
A \pkg{sf} point object.
}
\description{
Get a \pkg{sf} point with the location of the political powers for
each municipality (possibly the center of the municipality).

Note that this differs of the centroid of the boundaries of the
municipality, returned by \code{\link[=esp_get_munic]{esp_get_munic()}}.
}
\details{
\code{year} could be passed as a single year ("YYYY" format, as end of year) or
as a specific date ("YYYY-MM-DD" format). Historical information starts as
of 2005.

When using \code{region} you can use and mix names and NUTS codes (levels 1,
2 or 3), ISO codes (corresponding to level 2 or 3) or "cpro". See
\link{esp_codelist}

When calling a superior level (Province, Autonomous Community or NUTS1) ,
all the municipalities of that level would be added.
}
\section{About caching}{
You can set your \code{cache_dir} with \code{\link[=esp_set_cache_dir]{esp_set_cache_dir()}}.

Sometimes cached files may be corrupt. On that case, try re-downloading
the data setting \code{update_cache = TRUE}.

If you experience any problem on download, try to download the
corresponding .geojson file by any other method and save it on your
\code{cache_dir}. Use the option \code{verbose = TRUE} for debugging the API query.
}

\section{Displacing the Canary Islands}{
While \code{moveCAN} is useful for visualization, it would alter the actual
geographic position of the Canary Islands. When using the output for
spatial analysis or using tiles (e.g. with \code{\link[=esp_getTiles]{esp_getTiles()}} or
\code{\link[=addProviderEspTiles]{addProviderEspTiles()}})  this option should be set to \code{FALSE} in order to
get the actual coordinates, instead of the modified ones.
}

\examples{
\dontrun{
# This code compares centroids of municipalities against esp_get_capimun
# It also download tiles, make sure you are online

library(sf)

# Get shape
area <- esp_get_munic_siane(munic = "Valladolid", epsg = 3857)

# Area in km2
print(paste0(round(as.double(sf::st_area(area)) / 1000000, 2), " km2"))

# Extract centroid
centroid <- sf::st_centroid(area)
centroid$type <- "Centroid"

# Compare with capimun
capimun <- esp_get_capimun(munic = "Valladolid", epsg = 3857)
capimun$type <- "Capimun"

# Get a tile to check
tile <- esp_getTiles(area, "IGNBase.Todo", zoommin = 2)

# Join both point geometries
points <- rbind(
  centroid[, "type"],
  capimun[, "type"]
)


# Check on plot
library(ggplot2)
library(tidyterra)

ggplot(points) +
  geom_spatraster_rgb(data = tile) +
  geom_sf(data = area, fill = NA, color = "blue") +
  geom_sf(data = points, aes(fill = type), size = 5, shape = 21) +
  scale_fill_manual(values = c("green", "red")) +
  theme_void() +
  labs(title = "Centroid vs. capimun")
}
}
\seealso{
Other political: 
\code{\link{esp_codelist}},
\code{\link{esp_get_can_box}()},
\code{\link{esp_get_ccaa}()},
\code{\link{esp_get_comarca}()},
\code{\link{esp_get_country}()},
\code{\link{esp_get_gridmap}},
\code{\link{esp_get_munic}()},
\code{\link{esp_get_nuts}()},
\code{\link{esp_get_prov}()},
\code{\link{esp_get_simpl_prov}()}

Other municipalities: 
\code{\link{esp_get_munic}()},
\code{\link{esp_munic.sf}}
}
\concept{municipalities}
\concept{political}
