#' Compare results to NONMEM .phi
#'
#' @name vs_nonmem
#' @param x full path to a .phi file generated by NONMEM
#' @param mapbayr_phi results of mapbayr estimations, in the form of a tibble data.frame, typically obtained from `get_phi()`
#' @param nonmem_phi results of NONMEM estimations, in the form of a tibble data.frame, typically obtained from `read_nmphi()`
#' @param merged_phi merged results of estimations, typically obtained from `merge_phi()`
#' @param only_ETA filter the data with `type=="ETA"` before the plot (a logical, default is `TRUE`)
#'
#' @return
#'  - read_nmphi: a tibble data.frame with a format close to the original .phi file
#'  - merge_phi: a long-form tibble data.frame with results of mapbayr and NONMEM
#'  - plot_phi: a `ggplot2` object
#'
#' @details
#'
#' These functions were made to easily compare the results of mapbayr to NONMEM. For instance, it could be useful in the case of the transposition of a pre-existing NONMEM model into mapbayr. For this, you need to code your model in both mapbayr and NONMEM, and perform the MAP-Bayesian estimation on the **same dataset**. Ideally, the latter contains a substantial number of patients. NONMEM returns the estimations results into a .phi file.
#'
#' Use `read_nmphi()` to parse the NONMEM .phi file into a convenient tibble data.frame with the columns:
#'
#'  - `SUBJECT_NO`, `ID`: Subject identification.
#'  - `ETA1`, `ETA2`, ..., `ETAn`: Point estimates of eta.
#'  - `ETC1_1`, `ETC2_1`, `ETC2_2`, ...,  `ETCn_n`: Variance-covariance matrix of estimation.
#'  - `OBJ`: objective function value
#'
#' Use `get_phi()` to access to the estimations of mapbayr with the same "phi" format.
#'
#' Use `merge_phi()` to combine mapbayr and NONMEM "phi files" into a single long-form data.frame with the columns:
#'
#'  - `SUBJECT_NO`, `ID`: Subject identification.
#'  - `variable` name and its `type`: ETA (point estimate), VARIANCE (on-diagonal element of the matrix), COVARIANCE (off-diagonal), and OBJ.
#'  - `mapbayr` and `nonmem`: corresponding values
#'  - `adiff`: absolute difference between `mapbayr` and `nonmem` values.
#'
#' Use `plot_phi()` to graphically represent `adiff` *vs* `variable`. Alternatively, the table returned by `merge_phi()` is easy to play with in order to derive performance statistics or the graphical plot of your choice.
#'
#' @examples
#' library(mapbayr)
#' nmphi <- read_nmphi(system.file("nm001", "run001.phi", package = "mapbayr"))
#' est001 |>
#'   get_phi() |>
#'   merge_phi(nmphi) |>
#'   plot_phi()

#' @rdname vs_nonmem
#' @export
read_nmphi <- function(x){
  tab <- utils::read.table(x, skip = 1)
  names(tab) <- tab[1,]
  tab <- tab[-1,]
  tab <- tab %>%
    as_tibble() %>%
    rename_with(gsub, pattern = "[()]", replacement = "") %>%
    rename_with(gsub, pattern = ",", replacement = "_") %>%
    mutate(across(everything(), as.double))
  tab
}

is.variance <- function(x){
  str_extract_all(x, "\\d+") %>%
    map_lgl(~ .x[1]==.x[2])
}

#' @rdname vs_nonmem
#' @export
merge_phi <- function(mapbayr_phi, nonmem_phi){
  stopifnot(
    mapbayr_phi$SUBJECT_NO == nonmem_phi$SUBJECT_NO,
    mapbayr_phi$ID == nonmem_phi$ID,
    names(mapbayr_phi) == names(nonmem_phi)
  )

  full_join(
    pivot_longer(mapbayr_phi, cols = -c(.data$SUBJECT_NO, .data$ID), names_to = "variable", values_to = "mapbayr"),
    pivot_longer(nonmem_phi, cols = -c(.data$SUBJECT_NO, .data$ID), names_to = "variable", values_to = "nonmem"),
    by = c("SUBJECT_NO", "ID", "variable")
  ) %>%
    mutate(type = case_when(
      str_detect(variable, "ETA") ~ "ETA",
      str_detect(variable, "OBJ") ~ "OBJ",
      is.variance(variable) ~ "VARIANCE",
      TRUE ~ "COVARIANCE"
    ), .after = "variable") %>%
    mutate(adiff = abs(.data$mapbayr-.data$nonmem))

}

#' @rdname vs_nonmem
#' @export
plot_phi <- function(merged_phi, only_ETA = TRUE){
  dat <- merged_phi
  if(only_ETA) dat <- filter(dat, .data$type == "ETA")

  dat %>%
    ggplot(aes(.data$variable, .data$adiff, group = .data$ID)) +
    geom_line() +
    scale_y_log10(name = "absolute difference")
}
