% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/maq.R
\name{maq}
\alias{maq}
\title{Fit a Multi-Armed Qini.}
\usage{
maq(
  reward,
  cost,
  budget,
  DR.scores,
  target.with.covariates = TRUE,
  R = 0,
  paired.inference = TRUE,
  sample.weights = NULL,
  clusters = NULL,
  tie.breaker = NULL,
  num.threads = NULL,
  seed = 42
)
}
\arguments{
\item{reward}{A matrix of reward estimates.}

\item{cost}{A matrix of cost estimates. If the costs are the same for each unit, then this can also
be a \code{ncol(reward)}-length vector.}

\item{budget}{The maximum spend per unit to fit the MAQ path on.
Setting this to some large number, such as \code{sum(cost)}, will fit the path up to a maximum spend per unit
where each unit that is expected to benefit (that is, \code{reward} is positive) is treated.}

\item{DR.scores}{A matrix of rewards to evaluate the MAQ on. For valid statistical inference, the
reward and cost estimates should be obtained independently from this evaluation data.}

\item{target.with.covariates}{If TRUE (Default), then the optimal policy takes covariates into
account. If FALSE, then the optimal policy only takes the average reward and cost into account when
allocating treatment.}

\item{R}{Number of bootstrap replicates for computing standard errors. Default is 0
(only point estimates are computed).}

\item{paired.inference}{Whether to allow for paired tests with other cost curves fit on the same
evaluation data. If TRUE (Default) then the path of bootstrap replicates are stored in order to perform
paired comparisons that account for the correlation between curves evaluated on the same data. This
takes memory on the order of O(RnK) and requires the comparison objects to be fit with the same seed
and R values as well as the same number of samples.}

\item{sample.weights}{Weights given to an observation in estimation.
If NULL, each observation is given the same weight. Default is NULL.}

\item{clusters}{Vector of integers or factors specifying which cluster each observation corresponds to.
Default is NULL (ignored).}

\item{tie.breaker}{An optional permutation of the the integers 1 to nrow(rewards) used to
break potential ties in the optimal treatment allocation. If NULL, the ties are broken by
the lowest sample id (i.e. the sample appearing first in the data). Default is NULL.}

\item{num.threads}{Number of threads used in bootstrap replicates. By default, the number of threads
is set to the maximum hardware concurrency.}

\item{seed}{The seed of the C++ random number generator. Default is 42.}
}
\value{
A fit maq object.
}
\description{
Given estimated conditional average treatment effects (CATEs), and costs, fit a
generalization of the Qini curve that can be used to assess (and compare) treatment
allocation policies implied by the estimated CATEs and costs on a held-out evaluation set.
}
\examples{
\donttest{
if (require("grf", quietly = TRUE)) {

# Fit a CATE estimator on a training sample.
n <- 3000
p <- 5
X <- matrix(runif(n * p), n, p)
W <- as.factor(sample(c("A", "B", "C"), n, replace = TRUE))
Y <- X[, 1] + X[, 2] * (W == "B") + 1.5 * X[, 3] * (W == "C") + rnorm(n)
train <- sample(1:n, n/2)

tau.forest <- grf::multi_arm_causal_forest(X[train, ], Y[train], W[train])

# Predict CATEs on held out evaluation data.
test <- -train
tau.hat <- predict(tau.forest, X[test, ], drop = TRUE)$predictions

# Form cost estimates - the following are a toy example.
cost.hat <- cbind(X[test, 4] / 4, X[test, 5])

# Fit an evaluation forest to compute doubly robust scores on the test set.
eval.forest <- grf::multi_arm_causal_forest(X[test, ], Y[test], W[test])
DR.scores <- grf::get_scores(eval.forest, drop = TRUE)

# Fit a MAQ on evaluation data, using 200 bootstrap replicates for confidence intervals.
max.budget <- 1
mq <- maq(tau.hat, cost.hat, max.budget, DR.scores, R = 200)

# Plot the MAQ curve.
plot(mq)
legend("topleft", c("All arms", "95\% CI"), lty = c(1, 3))

# Get an estimate of optimal reward at a given spend per unit along with standard errors.
average_gain(mq, spend = 0.2)

# Get the optimal treatment allocation matrix at a given spend per unit.
pi.mat <- predict(mq, spend = 0.2)

# If the treatment randomization probabilities are known, then an alternative to
# evaluation via AIPW scores is to use inverse-propensity weighting (IPW).
W.hat <- rep(1/3, 3)
observed.W <- match(W, levels(W))
Y.mat <- matrix(0, length(W), nlevels(W))
Y.mat[cbind(seq_along(observed.W), observed.W)] <- Y
Y.ipw <- sweep(Y.mat, 2, W.hat, "/")
Y.ipw.test <- Y.ipw[test, -1] - Y.ipw[test, 1]

mq.ipw <- maq(tau.hat, cost.hat, max.budget, Y.ipw.test)
plot(mq.ipw, add = TRUE, col = 2)
legend("topleft", c("All arms", "95\% CI", "All arms (IPW)"), col = c(1, 1, 2), lty = c(1, 3, 1))

# Estimate some baseline policies.
# a) A policy that ignores covariates and only only takes the average reward/cost into account.
mq.avg <- maq(tau.hat, cost.hat, max.budget, DR.scores, target.with.covariates = FALSE, R = 200)

# b) A policy that only use arm 1.
mq.arm1 <- maq(tau.hat[, 1], cost.hat[, 1], max.budget, DR.scores[, 1], R = 200)

# c) A policy that only use arm 2.
mq.arm2 <- maq(tau.hat[, 2], cost.hat[, 2], max.budget, DR.scores[, 2], R = 200)

plot(mq, ci.args = NULL)
plot(mq.avg, col = 2, add = TRUE, ci.args = NULL)
plot(mq.arm1, col = 3, add = TRUE, ci.args = NULL)
plot(mq.arm2, col = 4, add = TRUE, ci.args = NULL)
legend("topleft", c("All arms (targeting)", "All arms (without targeting)", "Arm 1", "Arm 2"),
       col = 1:4, lty = 1)

# Estimate the value of employing all arms over a random allocation.
difference_gain(mq, mq.avg, spend = 0.2)

# Estimate the value of adding arm 1 to the optimal policy mix.
difference_gain(mq, mq.arm1, spend = 0.2)

# Estimate the value of adding arm 2 to the optimal policy mix.
difference_gain(mq, mq.arm2, spend = 0.2)

}
}
# Fit a MAQ on toy data.
n <- 1000
K <- 5
tau.hat <- matrix(1 + rnorm(n * K), n, K)
cost <- 0.05 + matrix(runif(n * K), n, K)
reward.eval <-  matrix(1 + rnorm(n * K), n, K)

max.budget <- sum(cost)
mq <- maq(tau.hat, cost, max.budget, reward.eval, R = 200)
print(mq)

# Plot the Qini curve.
plot(mq)

# Get an estimate of gain.
average_gain(mq, spend = 0.2)

# Get the underlying treatment allocation.
pi.mat <- predict(mq, spend = 0.2)

# Compare with another curve
qini1 <- maq(tau.hat[, 1], cost[, 1], max.budget, reward.eval[, 1], R = 200)
difference_gain(mq, qini1, spend = 0.2)

}
\references{
Sverdrup, Erik, Han Wu, Susan Athey, and Stefan Wager.
"Qini Curves for Multi-Armed Treatment Rules".
arXiv preprint arXiv:2306.11979, 2023.
}
