\name{mcgibbsit}
\alias{mcgibbsit}
\alias{print.mcgibbsit}
\title{Warnes and Raftery's MCGibbsit MCMC diagnostic}
\description{
  mcgibbsit provides an implementation of Warnes & Raftery's MCGibbsit
  run-length diagnostic for a set of (not-necessarily independent) MCMC
  samplers.  It combines the estimate error-bounding approach of Raftery
  and Lewis with the between chain variance verses within chain variance
  approach of Gelman and Rubin.  }

\usage{
mcgibbsit(data, q=0.025, r=0.0125, s=0.95, converge.eps=0.001,
          correct.cor=TRUE)
}
\arguments{
  \item{data}{an `mcmc' object.}
  \item{q}{quantile(s) to be estimated.}
  \item{r}{the desired margin of error of the estimate.}
  \item{s}{the probability of obtaining an estimate in the interval}
  \item{converge.eps}{Precision required for estimate of time to convergence.}
  \item{correct.cor}{should the between-chain correlation correction (R) be
    computed and applied.  Set to false for independent MCMC chains.}
}
\details{
  
  \code{mcgibbsit} computes the minimum run length \eqn{N_{min}}{Nmin},
  required burn in \eqn{M}, total run length \eqn{N}, run length
  inflation due to \emph{auto-correlation}, \eqn{I}, and the run length
  inflation due to \emph{between-chain} correlation, \eqn{R} for a set of
  exchangeable MCMC simulations which need not be independent.

  The normal usage is to perform an initial MCMC run of some
  pre-determined length (e.g., 300 iterations) for each of a set of
  \eqn{k} (e.g., \eqn{k=20}) MCMC samplers.  The output from these samplers
  is then read in to create an \code{mcmc.list} object and
  \code{mcgibbsit} is run specifying the desired accuracy of estimation
  for quantiles of interest.  This will return the minimum number of
  iterations to achieve the specified error bound.  The set of MCMC
  samplers is now run so that the total number of iterations exceeds
  this minimum, and \code{mcgibbsit} is again called.  This should
  continue until the number of iterations already complete is less than
  the minimum number computed by \code{mcgibbsit}.

  If the initial number of iterations in \code{data} is too
  small to perform the calculations, an error message is printed
  indicating the minimum pilot run length.

  The parameters \code{q}, \code{r}, \code{s}, \code{converge.eps}, and
  \code{correct.cor} can be supplied as vectors.  This will cause
  \code{mcgibbsit} to produce a list of results, with one element
  produced for each set of values.  I.e., setting \code{q=(0.025,0.975),
  r=(0.0125,0.005)} will yield a list containing two \code{mcgibbsit} objects,
  one computed with parameters \code{q=0.025, r=0.0125}, and
  the other with \code{q=0.975, r=0.005}.

}
\value{
  An \code{mcgibbsit} object with components

  \item{call}{parameters used to call 'mcgibbsit'}
  \item{params}{values of r, s, and q used}
  \item{resmatrix}{a matrix with 6 columns:
    \describe{
      \item{Nmin}{The minimum required sample size for a chain with no 
	correlation between consecutive samples. Positive 
	autocorrelation will increase the required sample
	size above this minimum value.}
      \item{M}{The number of `burn in' iterations to be discarded 
	(total over all chains).}
      \item{N}{The number of iterations after burn in required to
	estimate the quantile q to within an accuracy of
	+/- r with probability p (total over all chains).}
      \item{Total}{Overall number of iterations required (M + N).}
      \item{I}{An estimate (the `dependence factor') of the extent to
	which auto-correlation inflates the required sample
	size.  Values of `I' larger than 5 indicate strong
	autocorrelation which may be due to a poor choice of
	starting value, high posterior correlations, or
	`stickiness' of the MCMC algorithm.}
      \item{R}{An estimate of the extent to which between-chain
	correlation inflates the required sample size.  Large
	values of 'R' indicate that there is significant
	correlation between the chains and may be indicative
	of a lack of convergence or a poor multi-chain
	algorithm.}
    }
  }
  \item{nchains}{the number of MCMC chains in the data}
  \item{len}{the length of each chain}
}
\references{

  Warnes, G.W. (2004). The Normal Kernel Coupler: An adaptive MCMC
  method for efficiently sampling from multi-modal distributions (web
  site), \url{http://www.analytics.washington.edu/Zope/projects/MCMC/NKC/index.html}.
  
  Warnes, G.W. (2000).  Multi-Chain and Parallel Algorithms for Markov
  Chain Monte Carlo. Dissertation, Department of Biostatistics,
  University of Washington,

  Raftery, A.E. and Lewis, S.M. (1992).  One long run with diagnostics:
  Implementation strategies for Markov chain Monte Carlo. Statistical
  Science, 7, 493-497.

  Raftery, A.E. and Lewis, S.M. (1995).  The number of iterations,
  convergence diagnostics and generic Metropolis algorithms.  In
  Practical Markov Chain Monte Carlo (W.R. Gilks, D.J. Spiegelhalter and
  S. Richardson, eds.). London, U.K.: Chapman and Hall.

}
\author{Gregory R. Warnes \email{gregory\_r\_warnes@groton.pfizer.com}
    based on the the R function \code{raftery.diag} which is part of the
    'CODA' library.  \code{raftery.diag}, in turn, is based on the
    FORTRAN program `gibbsit' written by Steven Lewis which is available
    from the Statlib archive.  }

\seealso{ \code{\link{read.mcmc}} }

\examples{

## Create set of 20 example data files
set.seed(12234)
for(i in 1:20){
  x <- matrix(rnorm(1000),ncol=4)
  x[,4] <- x[,4] + 1/3 * (x[,1] + x[,2] + x[,3])
  colnames(x) <- c("alpha","beta","gamma", "nu")
  write.csv(x, file=paste("mcmc",i,"csv",sep="."), row.names=FALSE)
}

## Read the 20 data files and convert them into a single
## MCMC object.
data <- read.mcmc(20, "mcmc.#.csv", sep=",")

## Clean up after ourselves
unlink( paste("mcmc.",1:20,".csv", sep="") )

## Now run mcgibbsit...
mcgibbsit(data)

}
\keyword{models}
