% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcp.R
\encoding{UTF-8}
\name{mcp}
\alias{mcp}
\title{Fit Multiple Linear Segments And Their Change Points}
\usage{
mcp(
  model,
  data = NULL,
  prior = list(),
  family = gaussian(),
  par_x = NULL,
  sample = "post",
  cores = 1,
  chains = 3,
  iter = 3000,
  adapt = 1500,
  inits = NULL,
  jags_code = NULL
)
}
\arguments{
\item{model}{A list of formulas - one for each segment. The first formula
has the format \code{response ~ predictors} while the following formulas
have the format \code{response ~ changepoint ~ predictors}. The response
and change points can be omitted (\code{changepoint ~ predictors} assumes same
response. \code{~ predictors} assumes an intercept-only change point). The
following can be modeled:
\itemize{
\item \emph{Regular formulas:} e.g., \code{~ 1 + x}). \href{https://lindeloev.github.io/mcp/articles/formulas.html}{Read more}.
\item \emph{Extended formulas:}, e.g., \code{~ I(x^2) + exp(x) + sin(x)}. \href{https://lindeloev.github.io/mcp/articles/formulas.html}{Read more}.
\item \emph{Variance:} e.g., \code{~sigma(1)} for a simple variance change or
\code{~sigma(rel(1) + I(x^2))}) for more advanced variance structures. \href{https://lindeloev.github.io/mcp/articles/variance.html}{Read more}
\item \emph{Autoregression:} e.g., \code{~ar(1)} for a simple onset/change in AR(1) or
\verb{ar(2, 0 + x}) for an AR(2) increasing by \code{x}. \href{https://lindeloev.github.io/mcp/articles/arma.html}{Read more}
}}

\item{data}{Data.frame or tibble in long format.}

\item{prior}{Named list. Names are parameter names (\code{cp_i}, \code{int_i}, \code{xvar_i},
`sigma``) and the values are either
\itemize{
\item A JAGS distribution (e.g., \code{int_1 = "dnorm(0, 1) T(0,)"}) indicating a
conventional prior distribution. Uninformative priors based on data
properties are used where priors are not specified. This ensures good
parameter estimations, but it is a questionable for hypothesis testing.
\code{mcp} uses SD (not precision) for dnorm, dt, dlogis, etc. See
details. Change points are forced to be ordered through the priors using
truncation, except for uniform priors where the lower bound should be
greater than the previous change point, \code{dunif(cp_1, MAXX)}.
\item A numerical value (e.g., \code{int_1 = -2.1}) indicating a fixed value.
\item A model parameter name (e.g., \code{int_2 = "int_1"}), indicating that this parameter is shared -
typically between segments. If two varying effects are shared this way,
they will need to have the same grouping variable.
\item A scaled Dirichlet prior is supported for change points if they are all set to
\verb{cp_i = "dirichlet(N)} where \code{N} is the alpha for this change point and
\code{N = 1} is most often used. This prior is less informative about the
location of the change points than the default uniform prior, but it
samples less efficiently, so you will often need to set \code{iter} higher.
It is recommended for hypothesis testing and for the estimation of more
than 5 change points. \href{https://lindeloev.github.io/mcp/articles/priors.html}{Read more}.
}}

\item{family}{One of \code{gaussian()}, \code{binomial()}, \code{bernoulli()}, or \code{poission()}.
Only default link functions are currently supported.}

\item{par_x}{String (default: NULL). Only relevant if no segments contains
slope (no hint at what x is). Set this, e.g., par_x = "time".}

\item{sample}{One of
\itemize{
\item \code{"post"} (default): Sample the posterior.
\item \code{"prior"}: Sample only the prior. Plots, summaries, etc. will
use the prior. This is useful for prior predictive checks.
\item \code{"both"}: Sample both prior and posterior. Plots, summaries, etc.
will default to using the posterior. The prior only has effect when doing
Savage-Dickey density ratios in \code{\link{hypothesis}}.
\item \code{"none"} or \code{FALSE}: Do not sample. Returns an mcpfit
object without sample. This is useful if you only want to check
prior strings (fit$prior), the JAGS model (fit$jags_code), etc.
}}

\item{cores}{Positive integer or "all". Number of cores.
\itemize{
\item \code{1}: serial sampling (Default). \code{options(mc.cores = 3)} will dominate \code{cores = 1}
but not larger values of \code{cores}.
\item \verb{>1}: parallel sampling on this number of cores. Ideally set \code{chains}
to the same value. Note: \code{cores > 1} takes a few extra seconds the first
time it's called but subsequent calls will start sampling immediately.
\item \code{"all"}: use all cores but one and sets \code{chains} to the same value. This is
a convenient way to maximally use your computer's power.
}}

\item{chains}{Positive integer. Number of chains to run.}

\item{iter}{Positive integer. Number of post-warmup samples to draw. The number
of iterations per chain is \code{iter/chains}.}

\item{adapt}{Positive integer. Also sometimes called "burnin", this is the
number of samples used to reach convergence. Set lower for greater speed.
Set higher if the chains haven't converged yet or look at \href{https://lindeloev.github.io/mcp/articles/tips.html}{tips, tricks, and debugging}.}

\item{inits}{A list if initial values for the parameters. This can be useful
if a model fails to converge. Read more in \code{\link[rjags]{jags.model}}.
Defaults to \code{NULL}, i.e., no inits.}

\item{jags_code}{String. Pass JAGS code to \code{mcp} to use directly. This is useful if
you want to tweak the code in \code{fit$jags_code} and run it within the \code{mcp}
framework.}
}
\value{
An \code{\link{mcpfit}} object.
}
\description{
Given a model (a list of segment formulas), \code{mcp} infers the posterior
distributions of the parameters of each segment as well as the change points
between segments. \href{https://lindeloev.github.io/mcp/}{See more details and worked examples on the mcp website}.
All segments must regress on the same x-variable. Change
points are forced to be ordered using truncation of the priors. You can run
\code{fit = mcp(model, sample=FALSE)} to avoid sampling and the need for
data if you just want to get the priors (\code{fit$prior}), the JAGS code
\code{fit$jags_code}, or the R function to simulate data (\code{fit$simulate}).
}
\details{
Notes on priors:
\itemize{
\item Order restriction is automatically applied to cp_\* parameters using
truncation (e.g., \code{T(cp_1, )}) so that they are in the correct order on the
x-axis UNLESS you do it yourself. The one exception is for dunif
distributions where you have to do it as above.
\item In addition to the model parameters, \code{MINX} (minimum x-value), \code{MAXX}
(maximum x-value), \code{SDX} (etc...), \code{MINY}, \code{MAXY}, and \code{SDY}
are also available when you set priors. They are used to set uninformative
default priors.
\item Use SD when you specify priors for dt, dlogis, etc. JAGS uses precision
but \code{mcp} converts to precision under the hood via the sd_to_prec()
function. So you will see SDs in \code{fit$prior} but precision ($1/SD^2)
in \code{fit$jags_code}
}
}
\examples{
\donttest{
# Define the segments using formulas. A change point is estimated between each formula.
model = list(
  response ~ 1,  # Plateau in the first segment (int_1)
  ~ 0 + time,    # Joined slope (time_2) at cp_1
  ~ 1 + time     # Disjoined slope (int_3, time_3) at cp_2
)

# Fit it. The `ex_demo` dataset is included in mcp. Sample the prior too.
# options(mc.cores = 3)  # Uncomment to speed up sampling
ex_fit = mcp(model, data = ex_demo, sample = "both")

# See parameter estimates
summary(ex_fit)

# Visual inspection of the results
plot(ex_fit)  # Visualization of model fit/predictions
plot_pars(ex_fit)  # Parameter distributions
pp_check(ex_fit)  # Prior/Posterior predictive checks

# Test a hypothesis
hypothesis(ex_fit, "cp_1 > 10")

# Make predictions
fitted(ex_fit)
predict(ex_fit)
predict(ex_fit, newdata = data.frame(time = c(55.545, 80, 132)))

# Compare to a one-intercept-only model (no change points) with default prior
model_null = list(response ~ 1)
fit_null = mcp(model_null, data = ex_demo, par_x = "time")  # fit another model here
ex_fit$loo = loo(ex_fit)
fit_null$loo = loo(fit_null)
loo::loo_compare(ex_fit$loo, fit_null$loo)

# Inspect the prior. Useful for prior predictive checks.
summary(ex_fit, prior = TRUE)
plot(ex_fit, prior = TRUE)

# Show all priors. Default priors are added where you don't provide any
print(ex_fit$prior)

# Set priors and re-run
prior = list(
  int_1 = 15,
  time_2 = "dt(0, 2, 1) T(0, )",  # t-dist slope. Truncated to positive.
  cp_2 = "dunif(cp_1, 80)",    # change point to segment 2 > cp_1 and < 80.
  int_3 = "int_1"           # Shared intercept between segment 1 and 3
)

fit3 = mcp(model, data = ex_demo, prior = prior)

# Show the JAGS model
cat(ex_fit$jags_code)
}

}
\seealso{
\code{\link{get_segment_table}}
}
\author{
Jonas Kristoffer Lindeløv \email{jonas@lindeloev.dk}
}
