% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_land.R
\name{spwb_land_day}
\alias{spwb_land_day}
\alias{growth_land_day}
\title{One-day watershed simulations}
\usage{
spwb_land_day(
  r,
  sf,
  SpParams,
  meteo = NULL,
  date = NULL,
  local_control = medfate::defaultControl(soilDomains = "single"),
  watershed_control = default_watershed_control(),
  parallelize = FALSE,
  num_cores = parallel::detectCores() - 1,
  chunk_size = NULL,
  progress = TRUE
)

growth_land_day(
  r,
  sf,
  SpParams,
  meteo = NULL,
  date = NULL,
  local_control = medfate::defaultControl(soilDomains = "single"),
  watershed_control = default_watershed_control(),
  parallelize = FALSE,
  num_cores = parallel::detectCores() - 1,
  chunk_size = NULL,
  progress = TRUE
)
}
\arguments{
\item{r}{An object of class \code{\link[terra]{SpatRaster}}, defining the raster topology.}

\item{sf}{An object of class \code{\link[sf]{sf}} as described in \code{\link{spwb_land}}.}

\item{SpParams}{A data frame with species parameters (see \code{\link[medfate]{SpParamsMED}}).}

\item{meteo}{Input meteorological data (see \code{\link{spwb_spatial}} and details).}

\item{date}{A string with the date to be simulated.}

\item{local_control}{A list of control parameters (see \code{\link[medfate]{defaultControl}}) for function \code{\link[medfate]{spwb_day}} or \code{\link[medfate]{growth_day}}.}

\item{watershed_control}{A list of watershed control parameters (see \code{\link{default_watershed_control}}). Importantly, the sub-model used
for lateral water flows - either \enc{Francés}{Frances} et al. (2007) or \enc{Caviedes-Voullième}{Caviedes-Voullieme} et al. (2023) - is specified there.}

\item{parallelize}{Boolean flag to try parallelization (see details).}

\item{num_cores}{Integer with the number of cores to be used for parallel computation (by default it will use all clusters minus one).}

\item{chunk_size}{Integer indicating the size of chunks to be sent to different processes (by default, the number of spatial elements divided by the number of cores).}

\item{progress}{Boolean flag to display progress information for simulations.}
}
\value{
Functions \code{spwb_land_day} and \code{spwb_land_day} return a sf object:
\itemize{
\item{\code{geometry}: Spatial geometry.}
\item{\code{state}: A list of model input objects for each simulated stand.}
\item{\code{aquifer}: A numeric vector with the water volume in the aquifer of each cell.}
\item{\code{snowpack}: A numeric vector with the snowpack water equivalent volume of each cell.}
\item{\code{result}: A list of cell detailed results (only for those indicated in the input), with contents depending on the local model.}
\item{\code{outlet}: A logical vector indicating outlet cells.}
\item{\code{MinTemperature}: Minimum temperature (degrees Celsius).}
\item{\code{MaxTemperature}: Maximum temperature (degrees Celsius).}
\item{\code{PET}: Potential evapotranspiration (in mm).}
\item{\code{Rain}: Rainfall (in mm).}
\item{\code{Snow}: Snowfall (in mm).}
\item{\code{Snowmelt}: Snow melt (in mm).}
\item{\code{Interception}: Rainfall interception (in mm).}
\item{\code{NetRain}: Net rainfall, i.e. throughfall, (in mm).}
\item{\code{Infiltration}: The amount of water infiltrating into the soil (in mm).}
\item{\code{InfiltrationExcess}: The amount of water exceeding the soil infiltration capacity (in mm).}
\item{\code{SaturationExcess}: The amount of water that reaches the soil surface because of soil saturation (in mm).}
\item{\code{Runoff}: The amount of water exported via surface runoff (in mm).}
\item{\code{DeepDrainage}: The amount of water draining from soil to the aquifer via deep drainage (in mm).}
\item{\code{CapillarityRise}: Water entering the soil via capillarity rise (mm) from the water table.}
\item{\code{SoilEvaporation}: Bare soil evaporation (in mm).}
\item{\code{Transpiration}: Woody plant transpiration (in mm).}
\item{\code{HerbTranspiration}: Herbaceous transpiration (in mm).}
\item{\code{InterflowInput}: The amount of water that reaches the soil of the cell from adjacent cells via subsurface flow (in mm).}
\item{\code{InterflowOutput}: The amount of water that leaves the soil of the cell towards adjacent cells via subsurface flow (in mm).}
\item{\code{InterflowBalance}: The balance of water circulating via subsurface flow (in mm).}
\item{\code{BaseflowInput}: The amount of water that reaches the aquifer of the cell from adjacent cells via groundwater flow (in mm).}
\item{\code{BaseflowOutput}: The amount of water that leaves the aquifer of the cell towards adjacent cells via groundwater flow (in mm).}
\item{\code{BaseflowBalance}: The balance of water circulating via groundwater flow (in mm).}
\item{\code{AquiferExfiltration}: The amount of water of the cell that generates surface runoff due to the aquifer reaching the soil surface (in mm).}
}
}
\description{
Functions to perform one-day simulations on a watershed described by a set of connected grid cells.
\itemize{
\item{Function \code{spwb_land_day} implements a distributed hydrological model that simulates daily local water balance, from \code{\link[medfate]{spwb_day}},
on grid cells of a watershed while accounting for overland runoff, subsurface flow and groundwater flow between cells.}
\item{Function \code{growth_land_day} is similar to \code{spwb_land_day}, but includes daily local carbon balance, growth and mortality processes in grid cells,
provided by \code{\link[medfate]{growth_day}}.}
}
}
\details{
See details in \code{\link{spwb_land}}.
}
\examples{
# Load example watershed data after burnin period
data("example_watershed_burnin")

# Set request for daily model results in cells number 3, 6 (outlet) and 9
example_watershed_burnin$result_cell <- FALSE
example_watershed_burnin$result_cell[c(3,6,9)] <- TRUE

# Get bounding box to determine limits
b <- sf::st_bbox(example_watershed_burnin)
b

# Define a raster topology, using terra package, 
# with the same CRS as the watershed. In this example cells have 100 m side.
# Coordinates in the 'sf' object are assumed to be cell centers
r <-terra::rast(xmin = 401380, ymin = 4671820, xmax = 402880, ymax = 4672620, 
                nrow = 8, ncol = 15, crs = "epsg:32631")

# Load example meteo data frame from package meteoland
data("examplemeteo")
  
# Load default medfate parameters
data("SpParamsMED")
  
# Watershed control parameters (TETIS model; Frances et al. 2007)
ws_control <- default_watershed_control("tetis")

# Launch simulation 
date <- "2001-03-01"
sf_out <- spwb_land_day(r, example_watershed_burnin, SpParamsMED, examplemeteo, 
                        date = date, 
                        watershed_control = ws_control)

}
\references{
\enc{Francés}{Frances}, F., \enc{Vélez}{Velez}, J.I. & \enc{Vélez}{Velez}, J.J. (2007). Split-parameter structure for the automatic calibration of distributed hydrological models. Journal of Hydrology, 332, 226–240.

\enc{Caviedes-Voullième}{Caviedes-Voullieme}, D., \enc{Morales-Hernández}{Morales-Hernandez}, M., Norman, M.R. & Ogzen-Xian, I. (2023). SERGHEI (SERGHEI-SWE) v1.0: a performance-portable high-performance parallel-computing shallow-water solver for hydrology and environmental hydraulics. Geoscientific Model Development, 16, 977-1008.
}
\seealso{
\code{\link{default_watershed_control}},  \code{\link[medfate]{spwb_day}},  \code{\link[medfate]{growth_day}},
\code{\link{spwb_land}},
}
\author{
Miquel De \enc{Cáceres}{Caceres} Ainsa, CREAF.

Maria \enc{González-Sanchís}{Gonzalez-Sanchis}, Universitat Politecnica de Valencia.

Daniel \enc{Caviedes-Voullième}{Caviedes-Voullieme}, Forschungszentrum Julich.

Mario \enc{Morales-Hernández}{Morales-Hernandez}, Universidad de Zaragoza.
}
