% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mice.impute.quadratic.R
\name{mice.impute.quadratic}
\alias{mice.impute.quadratic}
\alias{quadratic}
\title{Imputation of quadratic terms}
\usage{
mice.impute.quadratic(y, ry, x, wy = NULL, quad.outcome = NULL, ...)
}
\arguments{
\item{y}{Vector to be imputed}

\item{ry}{Logical vector of length \code{length(y)} indicating the
the subset \code{y[ry]} of elements in \code{y} to which the imputation
model is fitted. The \code{ry} generally distinguishes the observed
(\code{TRUE}) and missing values (\code{FALSE}) in \code{y}.}

\item{x}{Numeric design matrix with \code{length(y)} rows with predictors for
\code{y}. Matrix \code{x} may have no missing values.}

\item{wy}{Logical vector of length \code{length(y)}. A \code{TRUE} value
indicates locations in \code{y} for which imputations are created.}

\item{quad.outcome}{The name of the outcome in the quadratic analysis as a
character string. For example, if the substantive model of interest is
\code{y ~ x + xx}, then \code{"y"} would be the \code{quad.outcome}}

\item{...}{Other named arguments.}
}
\value{
Vector with imputed data, same type as \code{y}, and of length
\code{sum(wy)}
}
\description{
Imputes incomplete variable that appears as both
main effect and quadratic effect in the complete-data model.
}
\details{
This function implements the "polynomial combination" method.
First, the polynomial
combination \eqn{Z = Y \beta_1 + Y^2 \beta_2} is formed.
\eqn{Z} is imputed by
predictive mean matching, followed by a decomposition of the imputed
data \eqn{Z}
into components \eqn{Y} and \eqn{Y^2}.
See Van Buuren (2012, pp. 139-141) and Vink
et al (2012) for more details. The method ensures that 1) the imputed data
for \eqn{Y} and \eqn{Y^2} are mutually consistent, and 2) that provides unbiased
estimates of the regression weights in a complete-data linear regression that
use both \eqn{Y} and \eqn{Y^2}.
}
\note{
There are two situations to consider. If only the linear term \code{Y}
is present in the data, calculate the quadratic term \code{YY} after
imputation. If both the linear term \code{Y} and the the quadratic term
\code{YY} are variables in the data, then first impute \code{Y} by calling
\code{mice.impute.quadratic()} on \code{Y}, and then impute \code{YY} by
passive imputation as \code{meth["YY"] <- "~I(Y^2)"}.  See example section
for details.  Generally, we would like \code{YY} to be present in the data if
we need to preserve quadratic relations between \code{YY} and any third
variables in the multivariate incomplete data that we might wish to impute.
}
\examples{
# Create Data
B1 <- .5
B2 <- .5
X <- rnorm(1000)
XX <- X^2
e <- rnorm(1000, 0, 1)
Y <- B1 * X + B2 * XX + e
dat <- data.frame(x = X, xx = XX, y = Y)

# Impose 25 percent MCAR Missingness
dat[0 == rbinom(1000, 1, 1 - .25), 1:2] <- NA

# Prepare data for imputation
ini <- mice(dat, maxit = 0)
meth <- c("quadratic", "~I(x^2)", "")
pred <- ini$pred
pred[, "xx"] <- 0

# Impute data
imp <- mice(dat, meth = meth, pred = pred, quad.outcome = "y")

# Pool results
pool(with(imp, lm(y ~ x + xx)))

# Plot results
stripplot(imp)
plot(dat$x, dat$xx, col = mdc(1), xlab = "x", ylab = "xx")
cmp <- complete(imp)
points(cmp$x[is.na(dat$x)], cmp$xx[is.na(dat$x)], col = mdc(2))
}
\seealso{
\code{\link{mice.impute.pmm}}
Van Buuren, S. (2018).
\href{https://stefvanbuuren.name/fimd/sec-knowledge.html#sec:quadratic}{\emph{Flexible Imputation of Missing Data. Second Edition.}}
Chapman & Hall/CRC. Boca Raton, FL.

Vink, G., van Buuren, S. (2013). Multiple Imputation of Squared Terms.
\emph{Sociological Methods & Research}, 42:598-607.

Other univariate imputation functions: 
\code{\link{mice.impute.cart}()},
\code{\link{mice.impute.lasso.logreg}()},
\code{\link{mice.impute.lasso.norm}()},
\code{\link{mice.impute.lasso.select.logreg}()},
\code{\link{mice.impute.lasso.select.norm}()},
\code{\link{mice.impute.lda}()},
\code{\link{mice.impute.logreg.boot}()},
\code{\link{mice.impute.logreg}()},
\code{\link{mice.impute.mean}()},
\code{\link{mice.impute.midastouch}()},
\code{\link{mice.impute.mnar.logreg}()},
\code{\link{mice.impute.mpmm}()},
\code{\link{mice.impute.norm.boot}()},
\code{\link{mice.impute.norm.nob}()},
\code{\link{mice.impute.norm.predict}()},
\code{\link{mice.impute.norm}()},
\code{\link{mice.impute.pmm}()},
\code{\link{mice.impute.polr}()},
\code{\link{mice.impute.polyreg}()},
\code{\link{mice.impute.rf}()},
\code{\link{mice.impute.ri}()}
}
\author{
Mingyang Cai and Gerko Vink
}
\concept{univariate imputation functions}
\keyword{datagen}
