% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mior.R
\name{mior}
\alias{mior}
\alias{mior.default}
\alias{mior.formula}
\alias{mior.mi_df}
\title{Fit MIOR model to the data}
\usage{
\method{mior}{default}(
  x,
  y,
  bags,
  cost = 1,
  cost_eta = 1,
  method = "qp-heuristic",
  weights = NULL,
  control = list(kernel = "linear", sigma = if (is.vector(x)) 1 else 1/ncol(x),
    max_step = 500, scale = TRUE, verbose = FALSE, time_limit = 60, option =
    c("corrected", "xiao")),
  ...
)

\method{mior}{formula}(formula, data, ...)

\method{mior}{mi_df}(x, ...)
}
\arguments{
\item{x}{A data.frame, matrix, or similar object of covariates, where each
row represents an instance. If a \code{mi_df} object is passed, \verb{y, bags} are
automatically extracted, and all other columns will be used as predictors.}

\item{y}{A numeric, character, or factor vector of bag labels for each
instance.  Must satisfy \code{length(y) == nrow(x)}. Suggest that one of the
levels is 1, '1', or TRUE, which becomes the positive class; otherwise, a
positive class is chosen and a message will be supplied.}

\item{bags}{A vector specifying which instance belongs to each bag.  Can be a
string, numeric, of factor.}

\item{cost}{The cost parameter in SVM. If \code{method = 'heuristic'}, this will
be fed to \code{kernlab::ksvm()}, otherwise it is similarly in internal
functions.}

\item{cost_eta}{The additional cost parameter in MIOR which controls how far
away the first and last separating hyperplanes are relative to other costs.}

\item{method}{The algorithm to use in fitting (default  \code{'heuristic'}).  When
\code{method = 'heuristic'}, which employs an algorithm similar to Andrews et
al. (2003). When \code{method = 'mip'}, the novel MIP method will be used.  When
\verb{method = 'qp-heuristic}, the heuristic algorithm is computed using the
dual SVM.  See details.}

\item{weights}{named vector, or \code{TRUE}, to control the weight of the cost
parameter for each possible y value.  Weights multiply against the cost
vector. If \code{TRUE}, weights are calculated based on inverse counts of
instances with given label, where we only count one positive instance per
bag. Otherwise, names must match the levels of \code{y}.}

\item{control}{list of additional parameters passed to the method that
control computation with the following components:
\itemize{
\item \code{kernel} either a character the describes the kernel ('linear' or
'radial') or a kernel matrix at the instance level.
\item \code{sigma} argument needed for radial basis kernel.
\item \code{max_step} argument used when \code{method = 'heuristic'}. Maximum steps of
iteration for the heuristic algorithm.
\item \code{scale} argument used for all methods. A logical for whether to rescale
the input before fitting.
\item \code{verbose} argument used when \code{method = 'mip'}. Whether to message output
to the console.
\item \code{time_limit} argument used when \code{method = 'mip'}. \code{FALSE}, or a time
limit (in seconds) passed to \code{gurobi()} parameters.  If \code{FALSE}, no time
limit is given.
\item \code{option} argument the controls the constraint calculation.  See details.
}}

\item{...}{Arguments passed to or from other methods.}

\item{formula}{a formula with specification \code{mi(y, bags) ~ x} which uses the
\code{mi} function to create the bag-instance structure. This argument is an
alternative to the \verb{x, y, bags} arguments, but requires the \code{data}
argument. See examples.}

\item{data}{If \code{formula} is provided, a data.frame or similar from which
formula elements will be extracted}
}
\value{
An object of class \code{mior}  The object contains at least the following
components:
\itemize{
\item \code{gurobi_fit}: A fit from model optimization that includes relevant
components.
\item \code{call_type}: A character indicating which method \code{misvm()} was called
with.
\item \code{features}: The names of features used in training.
\item \code{levels}: The levels of \code{y} that are recorded for future prediction.
\item \code{cost}: The cost parameter from function inputs.
\item \code{weights}: The calculated weights on the \code{cost} parameter.
\item \code{repr_inst}: The instances from positive bags that are selected to be
most representative of the positive instances.
\item \code{n_step}: If \code{method \%in\% c('heuristic', 'qp-heuristic')}, the total
steps used in the heuristic algorithm.
\item \code{x_scale}: If \code{scale = TRUE}, the scaling parameters for new predictions.
}
}
\description{
This function fits the MIOR model, proposed by Xiao Y, Liu B, and Hao Z
(2018) in "Multiple-instance Ordinal Regression".  MIOR is a modified SVM
framework with parallel, ordered hyperplanes where the error terms are based
only on the instance closest to a midpoint between hyperplanes.
}
\details{
Predictions (see \code{\link[=predict.mior]{predict.mior()}}) are determined by considering the smallest
distance from each point to the midpoint hyperplanes across all instances in
the bag.  The prediction corresponds to the hyperplane having such a minimal
distance.

It appears as though an error in Equation (12) persists to the dual form in
(21). A corrected version of this dual formulation can be used with
\code{control$option = 'corrected'}, or the formulation as written can be used
with \code{control$option = 'xiao'}.
}
\section{Methods (by class)}{
\itemize{
\item \code{default}: Method for data.frame-like objects

\item \code{formula}: Method for passing formula

\item \code{mi_df}: Method for \code{mi_df} objects, automatically handling bag
names, labels, and all covariates.
}}

\examples{
if (require(gurobi)) {
  set.seed(8)
  # make some data
  n <- 15
  X <- rbind(
    mvtnorm::rmvnorm(n/3, mean = c(4, -2, 0)),
    mvtnorm::rmvnorm(n/3, mean = c(0, 0, 0)),
    mvtnorm::rmvnorm(n/3, mean = c(-2, 1, 0))
  )
  score <- X \%*\% c(2, -1, 0)
  y <- as.numeric(cut(score, c(-Inf, quantile(score, probs = 1:2 / 3), Inf)))
  bags <- 1:length(y)

  # add in points outside boundaries
  X <- rbind(
    X,
    mvtnorm::rmvnorm(n, mean = c(6, -3, 0)),
    mvtnorm::rmvnorm(n, mean = c(-6, 3, 0))
  )
  y <- c(y, rep(-1, 2*n))
  bags <- rep(bags, 3)
  repr <- c(rep(1, n), rep(0, 2*n))

  y_bag <- classify_bags(y, bags, condense = FALSE)

  mdl1 <- mior(X, y_bag, bags)
  predict(mdl1, X, new_bags = bags)
}

}
\references{
Xiao, Y., Liu, B., & Hao, Z. (2017). Multiple-instance ordinal
regression. \emph{IEEE Transactions on Neural Networks and Learning Systems},
\emph{29}(9), 4398-4413. \doi{10.1109/TNNLS.2017.2766164}
}
\seealso{
\code{\link[=predict.misvm]{predict.misvm()}} for prediction on new data.
}
\author{
Sean Kent
}
