% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mirai.R
\name{mirai}
\alias{mirai}
\title{mirai (Evaluate Async)}
\usage{
mirai(.expr, ..., .args = list(), .timeout = NULL, .compute = "default")
}
\arguments{
\item{.expr}{an expression to evaluate asynchronously (of arbitrary length,
wrapped in \{ \} where necessary), \strong{or else} a pre-constructed
language object.}

\item{...}{(optional) \strong{either} named arguments (name = value pairs)
specifying objects referenced, but not defined, in \sQuote{.expr},
\strong{or} an environment containing such objects. See \sQuote{evaluation}
section below.}

\item{.args}{(optional) \strong{either} a named list specifying objects
referenced, but not defined, in \sQuote{.expr}, \strong{or} an environment
containing such objects. These objects will remain local to the evaluation
environment as opposed to those supplied in \sQuote{...} above - see
\sQuote{evaluation} section below.}

\item{.timeout}{[default NULL] for no timeout, or an integer value in
milliseconds. A mirai will resolve to an \sQuote{errorValue} 5 (timed out)
if evaluation exceeds this limit.}

\item{.compute}{[default 'default'] character value for the compute profile
to use (each compute profile has its own independent set of daemons).}
}
\value{
A \sQuote{mirai} object.
}
\description{
Evaluate an expression asynchronously in a new background R process or
persistent daemon (local or remote). This function will return immediately
with a \sQuote{mirai}, which will resolve to the evaluated result once
complete.
}
\details{
This function will return a \sQuote{mirai} object immediately.

The value of a mirai may be accessed at any time at \code{$data}, and if yet
to resolve, an \sQuote{unresolved} logical NA will be returned instead.

\code{\link{unresolved}} may be used on a mirai, returning TRUE if a
\sQuote{mirai} has yet to resolve and FALSE otherwise. This is suitable for
use in control flow statements such as \code{while} or \code{if}.

Alternatively, to call (and wait for) the result, use
\code{\link{call_mirai}} on the returned \sQuote{mirai}. This will block
until the result is returned.

Specify \sQuote{.compute} to send the mirai using a specific compute profile
(if previously created by \code{\link{daemons}}), otherwise leave as
\sQuote{default}.
}
\section{Evaluation}{


The expression \sQuote{.expr} will be evaluated in a separate R process in a
clean environment (not the global environment), consisting only of the
objects supplied to \sQuote{.args}, with the objects passed as \sQuote{...}
assigned to the global environment of that process.

As evaluation occurs in a clean environment, all undefined objects must be
supplied though \sQuote{...} and/or \sQuote{.args}, including self-defined
functions. Functions from a package should use namespaced calls such as
\code{mirai::mirai()}, or else the package should be loaded beforehand as
part of \sQuote{.expr}.

For evaluation to occur \emph{as if} in your global environment, supply
objects to \sQuote{...} rather than \sQuote{.args}, e.g. for free variables
or helper functions defined in function bodies, as scoping rules may
otherwise prevent them from being found.
}

\section{Timeouts}{


Specifying the \sQuote{.timeout} argument ensures that the mirai always
resolves. However, the task may not have completed and still be ongoing in
the daemon process. Use \code{\link{stop_mirai}} instead to explicitly stop
and interrupt a task.
}

\section{Errors}{


If an error occurs in evaluation, the error message is returned as a
character string of class \sQuote{miraiError} and \sQuote{errorValue}.
\code{\link{is_mirai_error}} may be used to test for this. The elements of
the original condition are accessible via \code{$} on the error object. A
stack trace is also available at \code{$stack.trace}.

If a daemon crashes or terminates unexpectedly during evaluation, an
\sQuote{errorValue} 19 (Connection reset) is returned.

\code{\link{is_error_value}} tests for all error conditions including
\sQuote{mirai} errors, interrupts, and timeouts.
}

\examples{
if (interactive()) {
# Only run examples in interactive R sessions

# specifying objects via '...'
n <- 3
m <- mirai(x + y + 2, x = 2, y = n)
m
m$data
Sys.sleep(0.2)
m$data

# passing the calling environment to '...'
df1 <- data.frame(a = 1, b = 2)
df2 <- data.frame(a = 3, b = 1)
m <- mirai(as.matrix(rbind(df1, df2)), environment(), .timeout = 1000)
m[]

# using unresolved()
m <- mirai(
  {
    res <- rnorm(n)
    res / rev(res)
  },
  n = 1e6
)
while (unresolved(m)) {
  cat("unresolved\n")
  Sys.sleep(0.1)
}
str(m$data)

# evaluating scripts using source() in '.expr'
n <- 10L
file <- tempfile()
cat("r <- rnorm(n)", file = file)
m <- mirai({source(file); r}, file = file, n = n)
call_mirai(m)$data
unlink(file)

# use source(local = TRUE) when passing in local variables via '.args'
n <- 10L
file <- tempfile()
cat("r <- rnorm(n)", file = file)
m <- mirai({source(file, local = TRUE); r}, .args = list(file = file, n = n))
call_mirai(m)$data
unlink(file)

# passing a language object to '.expr' and a named list to '.args'
expr <- quote(a + b + 2)
args <- list(a = 2, b = 3)
m <- mirai(.expr = expr, .args = args)
collect_mirai(m)

}

}
