% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/itemfit.R
\name{itemfit}
\alias{itemfit}
\title{Item fit statistics}
\usage{
itemfit(x, which.items = 1:extract.mirt(x, "nitems"), S_X2 = TRUE,
  Zh = FALSE, X2 = FALSE, G2 = FALSE, infit = FALSE, group.bins = 10,
  group.size = NA, group.fun = mean, mincell = 1, mincell.X2 = 2,
  S_X2.tables = FALSE, empirical.plot = NULL, empirical.CI = 0.95,
  empirical.table = NULL, method = "EAP", Theta = NULL, impute = 0,
  digits = 4, par.strip.text = list(cex = 0.7),
  par.settings = list(strip.background = list(col = "#9ECAE1"), strip.border =
  list(col = "black")), ...)
}
\arguments{
\item{x}{a computed model object of class \code{SingleGroupClass},
\code{MultipleGroupClass}, or \code{DiscreteClass}}

\item{which.items}{an integer vector indicating which items to test for fit.
Default tests all possible items}

\item{S_X2}{logical; calculate the S_X2 statistic?}

\item{Zh}{logical; calculate Zh?}

\item{X2}{logical; calculate the X2 statistic for unidimensional models?}

\item{G2}{logical; calculate the G2 statistic for unidimensional models?}

\item{infit}{logical; calculate the infit/outfit values for unidimensional Rasch models? If the models
are not from the Rasch family then this will be ignored}

\item{group.bins}{the number of bins to use when \code{X2 = TRUE}. For example,
setting \code{group.bins = 10} will will compute Yen's (1981) Q1 statistic}

\item{group.size}{approximate size of each group to be used in calculating the \eqn{\chi^2}
statistic. The default \code{NA}
disables this command and instead uses the \code{group.bins} input to try and construct
equally sized bins}

\item{group.fun}{function used when \code{X2} or \code{G2} are computed. Determines the central
tendancy measure within each partitioned group. E.g., setting \code{group.fun = median} will
obtain the median of each respective ability estimate in each subgroup (this is what was used
by Bock, 1972)}

\item{mincell}{the minimum expected cell size to be used in the S-X2 computations. Tables will be
collapsed across items first if polytomous, and then across scores if necessary}

\item{mincell.X2}{the minimum expected cell size to be used in the X2 computations. Tables will be
collapsed if polytomous, however if this condition can not be met then the group block will
be ommited in the computations}

\item{S_X2.tables}{logical; return the tables in a list format used to compute the S-X2 stats?}

\item{empirical.plot}{a single numeric value or character of the item name indicating which
item to plot (via \code{itemplot}) and overlay with the empirical \eqn{\theta} groupings (see
\code{empirical.CI}). Useful for plotting the expected bins after passing \code{X2 = TRUE} or
\code{G2 = TRUE}}

\item{empirical.CI}{a numeric value indicating the width of the empirical confidence interval
ranging between 0 and 1 (default of 0 plots not interval). For example, a 95% confidence
interval would be plotted when \code{empirical.CI = .95}. Only applicable to dichotomous items}

\item{empirical.table}{a single numeric value or character of the item name indicating which
item table of expected values should be returned. Useful for plotting the expected bins after
passing \code{X2 = TRUE} or \code{G2 = TRUE}}

\item{method}{type of factor score estimation method. See \code{\link{fscores}} for more detail}

\item{Theta}{a matrix of factor scores for each person used for statistics that require
empirical estimates. If supplied, arguments typically passed to \code{fscores()} will be
ignored and these values will be used instead. Also required when estimating statistics
with missing data via imputation}

\item{impute}{a number indicating how many imputations to perform (passed to
\code{\link{imputeMissing}}) when there are missing data present.
Will return a data.frame object with the mean estimates
of the stats and their imputed standard deviations}

\item{digits}{number of digits to round result to. Default is 4}

\item{par.strip.text}{plotting argument passed to \code{\link{lattice}}}

\item{par.settings}{plotting argument passed to \code{\link{lattice}}}

\item{...}{additional arguments to be passed to \code{fscores()} and \code{\link{lattice}}}
}
\description{
\code{itemfit} calculates the Zh values from Drasgow, Levine and Williams (1985),
\eqn{\chi^2} and \eqn{G^2} values for unidimensional models, and S-X2 statistics for unidimensional and
multidimensional models (Kang & Chen, 2007; Orlando & Thissen, 2000).
For Rasch, partial credit, and rating scale models infit and outfit statistics are
also produced. Poorly fitting items should be inspected with the empirical plots/tables
for unidimensional models, otherwise \code{\link{itemGAM}} can be used to diagnose
where the functional form of the IRT model was misspecified, or models can be refit using
more flexible semi-parametric response models (e.g., \code{itemtype = 'spline'}).
For discrete models, only the S-X2 statistic will be calculated.
}
\examples{

\dontrun{

P <- function(Theta){exp(Theta^2 * 1.2 - 1) / (1 + exp(Theta^2 * 1.2 - 1))}

#make some data
set.seed(1234)
a <- matrix(rlnorm(20, meanlog=0, sdlog = .1),ncol=1)
d <- matrix(rnorm(20),ncol=1)
Theta <- matrix(rnorm(2000))
items <- rep('dich', 20)
ps <- P(Theta)
baditem <- numeric(2000)
for(i in 1:2000)
   baditem[i] <- sample(c(0,1), 1, prob = c(1-ps[i], ps[i]))
data <- cbind(simdata(a,d, 2000, items, Theta=Theta), baditem=baditem)

x <- mirt(data, 1)
raschfit <- mirt(data, 1, itemtype='Rasch')
fit <- itemfit(x)
fit

itemfit(x, X2=TRUE)
itemfit(x, group.bins=15, empirical.plot = 1) #empirical item plot with 15 points
itemfit(x, group.bins=15, empirical.plot = 21)

#empirical tables
itemfit(x, empirical.table=1)
itemfit(x, empirical.table=21)

#infit/outfit statistics. method='ML' agrees better with eRm package
itemfit(raschfit, method = 'ML', infit = TRUE) #infit and outfit stats

#same as above, but inputting ML estimates instead
Theta <- fscores(raschfit, method = 'ML')
itemfit(raschfit, Theta=Theta, infit = TRUE)

# fit a new more flexible model for the mis-fitting item
itemtype <- c(rep('2PL', 20), 'spline')
x2 <- mirt(data, 1, itemtype=itemtype)
itemfit(x2)
itemplot(x2, 21)
anova(x2, x)

#------------------------------------------------------------

#similar example to Kang and Chen 2007
a <- matrix(c(.8,.4,.7, .8, .4, .7, 1, 1, 1, 1))
d <- matrix(rep(c(2.0,0.0,-1,-1.5),10), ncol=4, byrow=TRUE)
dat <- simdata(a,d,2000, itemtype = rep('graded', 10))
head(dat)

mod <- mirt(dat, 1)
itemfit(mod)
itemfit(mod, X2 = TRUE)
itemfit(mod, empirical.plot = 1)

# collapsed tables (see mincell.X2) for X2 and G2
itemfit(mod, empirical.table = 1)

mod2 <- mirt(dat, 1, 'Rasch')
itemfit(mod2, infit = TRUE)

#massive list of tables
tables <- itemfit(mod, S_X2.tables = TRUE)

#observed and expected total score patterns for item 1 (post collapsing)
tables$O[[1]]
tables$E[[1]]

# fit stats with missing data (run in parallel using all cores)
data[sample(1:prod(dim(data)), 500)] <- NA
raschfit <- mirt(data, 1, itemtype='Rasch')

mirtCluster() # run in parallel
itemfit(raschfit, impute = 10, infit = TRUE)

#alternative route: use only valid data, and create a model with the previous parameter estimates
data2 <- na.omit(data)
raschfit2 <- mirt(data2, 1, itemtype = 'Rasch', pars=mod2values(raschfit), TOL=NaN)
itemfit(raschfit2, infit = TRUE)

# note that X2 and G2 do not require complete datasets
itemfit(raschfit, X2=TRUE, G2 = TRUE, S_X2=FALSE)
itemfit(raschfit, empirical.plot=1)
itemfit(raschfit, empirical.table=1)

}

}
\author{
Phil Chalmers \email{rphilip.chalmers@gmail.com}
}
\references{
Bock, R. D. (1972). Estimating item parameters and latent ability when responses are scored
in two or more nominal categories. Psychometrika, 37, 29-51.

Drasgow, F., Levine, M. V., & Williams, E. A. (1985). Appropriateness measurement with
polychotomous item response models and standardized indices.
\emph{British Journal of Mathematical and Statistical Psychology, 38}, 67-86.

Kang, T. & Chen, Troy, T. (2007). An investigation of the performance of the generalized
S-X2 item-fit index for polytomous IRT models. ACT

McKinley, R., & Mills, C. (1985). A comparison of several goodness-of-fit statistics.
Applied Psychological Measurement, 9, 49-57.

Orlando, M. & Thissen, D. (2000). Likelihood-based item fit indices for dichotomous item
response theory models. \emph{Applied Psychological Measurement, 24}, 50-64.

Reise, S. P. (1990). A comparison of item- and person-fit methods of assessing model-data fit
in IRT. \emph{Applied Psychological Measurement, 14}, 127-137.

Wright B. D. & Masters, G. N. (1982). \emph{Rating scale analysis}. MESA Press.
}
\seealso{
\code{\link{personfit}}, \code{\link{itemGAM}}
}
\keyword{fit}
\keyword{item}

