% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mkinfit.R
\name{mkinfit}
\alias{mkinfit}
\title{Fit a kinetic model to data with one or more state variables}
\usage{
mkinfit(
  mkinmod,
  observed,
  parms.ini = "auto",
  state.ini = "auto",
  err.ini = "auto",
  fixed_parms = NULL,
  fixed_initials = names(mkinmod$diffs)[-1],
  from_max_mean = FALSE,
  solution_type = c("auto", "analytical", "eigen", "deSolve"),
  method.ode = "lsoda",
  use_compiled = "auto",
  control = list(eval.max = 300, iter.max = 200),
  transform_rates = TRUE,
  transform_fractions = TRUE,
  quiet = FALSE,
  atol = 1e-08,
  rtol = 1e-10,
  error_model = c("const", "obs", "tc"),
  error_model_algorithm = c("auto", "d_3", "direct", "twostep", "threestep",
    "fourstep", "IRLS", "OLS"),
  reweight.tol = 1e-08,
  reweight.max.iter = 10,
  trace_parms = FALSE,
  test_residuals = FALSE,
  ...
)
}
\arguments{
\item{mkinmod}{A list of class \link{mkinmod}, containing the kinetic
model to be fitted to the data, or one of the shorthand names ("SFO",
"FOMC", "DFOP", "HS", "SFORB", "IORE"). If a shorthand name is given, a
parent only degradation model is generated for the variable with the
highest value in \code{observed}.}

\item{observed}{A dataframe with the observed data.  The first column called
"name" must contain the name of the observed variable for each data point.
The second column must contain the times of observation, named "time".
The third column must be named "value" and contain the observed values.
Zero values in the "value" column will be removed, with a warning, in
order to avoid problems with fitting the two-component error model. This
is not expected to be a problem, because in general, values of zero are
not observed in degradation data, because there is a lower limit of
detection.}

\item{parms.ini}{A named vector of initial values for the parameters,
including parameters to be optimised and potentially also fixed parameters
as indicated by \code{fixed_parms}.  If set to "auto", initial values for
rate constants are set to default values.  Using parameter names that are
not in the model gives an error.

It is possible to only specify a subset of the parameters that the model
needs. You can use the parameter lists "bparms.ode" from a previously
fitted model, which contains the differential equation parameters from
this model.  This works nicely if the models are nested. An example is
given below.}

\item{state.ini}{A named vector of initial values for the state variables of
the model. In case the observed variables are represented by more than one
model variable, the names will differ from the names of the observed
variables (see \code{map} component of \link{mkinmod}). The default
is to set the initial value of the first model variable to the mean of the
time zero values for the variable with the maximum observed value, and all
others to 0.  If this variable has no time zero observations, its initial
value is set to 100.}

\item{err.ini}{A named vector of initial values for the error model
parameters to be optimised.  If set to "auto", initial values are set to
default values.  Otherwise, inital values for all error model parameters
must be given.}

\item{fixed_parms}{The names of parameters that should not be optimised but
rather kept at the values specified in \code{parms.ini}. Alternatively,
a named numeric vector of parameters to be fixed, regardless of the values
in parms.ini.}

\item{fixed_initials}{The names of model variables for which the initial
state at time 0 should be excluded from the optimisation. Defaults to all
state variables except for the first one.}

\item{from_max_mean}{If this is set to TRUE, and the model has only one
observed variable, then data before the time of the maximum observed value
(after averaging for each sampling time) are discarded, and this time is
subtracted from all remaining time values, so the time of the maximum
observed mean value is the new time zero.}

\item{solution_type}{If set to "eigen", the solution of the system of
differential equations is based on the spectral decomposition of the
coefficient matrix in cases that this is possible. If set to "deSolve", a
numerical \link[deSolve:ode]{ode solver from package deSolve} is used. If
set to "analytical", an analytical solution of the model is used. This is
only implemented for relatively simple degradation models.  The default is
"auto", which uses "analytical" if possible, otherwise "deSolve" if a
compiler is present, and "eigen" if no compiler is present and the model
can be expressed using eigenvalues and eigenvectors.}

\item{method.ode}{The solution method passed via \code{\link[=mkinpredict]{mkinpredict()}}
to \code{\link[deSolve:ode]{deSolve::ode()}} in case the solution type is "deSolve". The default
"lsoda" is performant, but sometimes fails to converge.}

\item{use_compiled}{If set to \code{FALSE}, no compiled version of the
\link{mkinmod} model is used in the calls to \code{\link[=mkinpredict]{mkinpredict()}} even if a compiled
version is present.}

\item{control}{A list of control arguments passed to \code{\link[stats:nlminb]{stats::nlminb()}}.}

\item{transform_rates}{Boolean specifying if kinetic rate constants should
be transformed in the model specification used in the fitting for better
compliance with the assumption of normal distribution of the estimator. If
TRUE, also alpha and beta parameters of the FOMC model are
log-transformed, as well as k1 and k2 rate constants for the DFOP and HS
models and the break point tb of the HS model.  If FALSE, zero is used as
a lower bound for the rates in the optimisation.}

\item{transform_fractions}{Boolean specifying if formation fractions
should be transformed in the model specification used in the fitting for
better compliance with the assumption of normal distribution of the
estimator. The default (TRUE) is to do transformations. If TRUE,
the g parameter of the DFOP model is also transformed. Transformations
are described in \link{transform_odeparms}.}

\item{quiet}{Suppress printing out the current value of the negative
log-likelihood after each improvement?}

\item{atol}{Absolute error tolerance, passed to \code{\link[deSolve:ode]{deSolve::ode()}}. Default
is 1e-8, which is lower than the default in the \code{\link[deSolve:lsoda]{deSolve::lsoda()}}
function which is used per default.}

\item{rtol}{Absolute error tolerance, passed to \code{\link[deSolve:ode]{deSolve::ode()}}. Default
is 1e-10, much lower than in \code{\link[deSolve:lsoda]{deSolve::lsoda()}}.}

\item{error_model}{If the error model is "const", a constant standard
deviation is assumed.

If the error model is "obs", each observed variable is assumed to have its
own variance.

If the error model is "tc" (two-component error model), a two component
error model similar to the one described by Rocke and Lorenzato (1995) is
used for setting up the likelihood function.  Note that this model
deviates from the model by Rocke and Lorenzato, as their model implies
that the errors follow a lognormal distribution for large values, not a
normal distribution as assumed by this method.}

\item{error_model_algorithm}{If "auto", the selected algorithm depends on
the error model.  If the error model is "const", unweighted nonlinear
least squares fitting ("OLS") is selected. If the error model is "obs", or
"tc", the "d_3" algorithm is selected.

The algorithm "d_3" will directly minimize the negative log-likelihood
and independently also use the three step algorithm described below.
The fit with the higher likelihood is returned.

The algorithm "direct" will directly minimize the negative log-likelihood.

The algorithm "twostep" will minimize the negative log-likelihood after an
initial unweighted least squares optimisation step.

The algorithm "threestep" starts with unweighted least squares, then
optimizes only the error model using the degradation model parameters
found, and then minimizes the negative log-likelihood with free
degradation and error model parameters.

The algorithm "fourstep" starts with unweighted least squares, then
optimizes only the error model using the degradation model parameters
found, then optimizes the degradation model again with fixed error model
parameters, and finally minimizes the negative log-likelihood with free
degradation and error model parameters.

The algorithm "IRLS" (Iteratively Reweighted Least Squares) starts with
unweighted least squares, and then iterates optimization of the error
model parameters and subsequent optimization of the degradation model
using those error model parameters, until the error model parameters
converge.}

\item{reweight.tol}{Tolerance for the convergence criterion calculated from
the error model parameters in IRLS fits.}

\item{reweight.max.iter}{Maximum number of iterations in IRLS fits.}

\item{trace_parms}{Should a trace of the parameter values be listed?}

\item{test_residuals}{Should the residuals be tested for normal distribution?}

\item{\dots}{Further arguments that will be passed on to
\code{\link[deSolve:ode]{deSolve::ode()}}.}
}
\value{
A list with "mkinfit" in the class attribute.
}
\description{
This function maximises the likelihood of the observed data using the Port
algorithm \code{\link[stats:nlminb]{stats::nlminb()}}, and the specified initial or fixed
parameters and starting values.  In each step of the optimisation, the
kinetic model is solved using the function \code{\link[=mkinpredict]{mkinpredict()}}, except
if an analytical solution is implemented, in which case the model is solved
using the degradation function in the \link{mkinmod} object. The
parameters of the selected error model are fitted simultaneously with the
degradation model parameters, as both of them are arguments of the
likelihood function.
}
\details{
Per default, parameters in the kinetic models are internally transformed in
order to better satisfy the assumption of a normal distribution of their
estimators.
}
\note{
When using the "IORE" submodel for metabolites, fitting with
"transform_rates = TRUE" (the default) often leads to failures of the
numerical ODE solver. In this situation it may help to switch off the
internal rate transformation.
}
\examples{

# Use shorthand notation for parent only degradation
fit <- mkinfit("FOMC", FOCUS_2006_C, quiet = TRUE)
summary(fit)

# One parent compound, one metabolite, both single first order.
# We remove zero values from FOCUS dataset D in order to avoid warnings
FOCUS_D <- subset(FOCUS_2006_D, value != 0)
# Use mkinsub for convenience in model formulation. Pathway to sink included per default.
SFO_SFO <- mkinmod(
  parent = mkinsub("SFO", "m1"),
  m1 = mkinsub("SFO"))

# Fit the model quietly to the FOCUS example dataset D using defaults
fit <- mkinfit(SFO_SFO, FOCUS_D, quiet = TRUE)
plot_sep(fit)
# As lower parent values appear to have lower variance, we try an alternative error model
fit.tc <- mkinfit(SFO_SFO, FOCUS_D, quiet = TRUE, error_model = "tc")
# This avoids the warning, and the likelihood ratio test confirms it is preferable
lrtest(fit.tc, fit)
# We can also allow for different variances of parent and metabolite as error model
fit.obs <- mkinfit(SFO_SFO, FOCUS_D, quiet = TRUE, error_model = "obs")
# The two-component error model has significantly higher likelihood
lrtest(fit.obs, fit.tc)
parms(fit.tc)
endpoints(fit.tc)

# We can show a quick (only one replication) benchmark for this case, as we
# have several alternative solution methods for the model. We skip
# uncompiled deSolve, as it is so slow. More benchmarks are found in the
# benchmark vignette
\dontrun{
if(require(rbenchmark)) {
  benchmark(replications = 1, order = "relative", columns = c("test", "relative", "elapsed"),
    deSolve_compiled = mkinfit(SFO_SFO, FOCUS_D, quiet = TRUE, error_model = "tc",
      solution_type = "deSolve", use_compiled = TRUE),
    eigen = mkinfit(SFO_SFO, FOCUS_D, quiet = TRUE, error_model = "tc",
      solution_type = "eigen"),
    analytical = mkinfit(SFO_SFO, FOCUS_D, quiet = TRUE, error_model = "tc",
      solution_type = "analytical"))
}
}

# Use stepwise fitting, using optimised parameters from parent only fit, FOMC-SFO
\dontrun{
FOMC_SFO <- mkinmod(
  parent = mkinsub("FOMC", "m1"),
  m1 = mkinsub("SFO"))
fit.FOMC_SFO <- mkinfit(FOMC_SFO, FOCUS_D, quiet = TRUE)
# Again, we get a warning and try a more sophisticated error model
fit.FOMC_SFO.tc <- mkinfit(FOMC_SFO, FOCUS_D, quiet = TRUE, error_model = "tc")
# This model has a higher likelihood, but not significantly so
lrtest(fit.tc, fit.FOMC_SFO.tc)
# Also, the missing standard error for log_beta and the t-tests for alpha
# and beta indicate overparameterisation
summary(fit.FOMC_SFO.tc, data = FALSE)

# We can easily use starting parameters from the parent only fit (only for illustration)
fit.FOMC = mkinfit("FOMC", FOCUS_2006_D, quiet = TRUE, error_model = "tc")
fit.FOMC_SFO <- mkinfit(FOMC_SFO, FOCUS_D, quiet = TRUE,
  parms.ini = fit.FOMC$bparms.ode, error_model = "tc")
}
}
\references{
Rocke DM and Lorenzato S (1995) A two-component model
for measurement error in analytical chemistry. \emph{Technometrics} 37(2), 176-184.

Ranke J and Meinecke S (2019) Error Models for the Kinetic Evaluation of Chemical
Degradation Data. \emph{Environments} 6(12) 124
\doi{10.3390/environments6120124}.
}
\seealso{
\link{summary.mkinfit}, \link{plot.mkinfit}, \link{parms} and \link{lrtest}.

Comparisons of models fitted to the same data can be made using
\code{\link{AIC}} by virtue of the method \code{\link{logLik.mkinfit}}.

Fitting of several models to several datasets in a single call to
\code{\link{mmkin}}.
}
\author{
Johannes Ranke
}
