
#' @title carries out preimputation
#' @description instead of replacing missing data with mean and mode, a smarter
#'              start-point would be to use fast imputation algorithms and then
#'              optimize the imputed dataset with mlim. this procedure usually
#'              requires less iterations and will savea lot of computation
#'              resources.
# @importFrom VIM kNN
#' @importFrom missRanger missRanger imputeUnivariate
# @importFrom missForest missForest
#' @param data data.frame with missing values
#' @param preimpute character. specify the algorithm for preimputation. the
#'                  supported options are "RF" (Random Forest) and "mm"
#'                  (mean-mode replacement). the default is "RF", which carries
#'                  a parallel random forest imputation, using all the CPUs available.
#'                  the other alternative is "mm" which performs mean/mode
#'                  imputation.
#' @param seed integer. specify the random generator seed
#' @return imputed data.frame
#' @author E. F. Haghish
#' @examples
#' \donttest{
#' data(iris)
#'
#' # add 10% stratified missing values to one factor variable
#' irisNA <- iris
#' irisNA$Species <- mlim.na(irisNA$Species, p = 0.1, stratify = TRUE, seed = 2022)
#'
#' # run the default random forest preimputation
#' MLIM <- mlim.preimpute(irisNA)
#' mlim.error(MLIM, irisNA, iris)

#' }
#' @export

mlim.preimpute <- function(data, preimpute = "RF", seed = NULL) {

  #if (tolower(preimpute) == "knn") {
  #  set.seed(seed)
  #  data <- VIM::kNN(data, imp_var=FALSE)
  #  if (!is.null(report)) md.log("kNN preimputation is done", date=debug, time=debug, trace=FALSE)
  #}
  if (tolower(preimpute) == "rf") {
    message("\nRandom Forest preimputation in progress...\n")
    data <- missRanger::missRanger(data, num.trees=500, mtry=1,
                                   verbose = 0, returnOOB=TRUE, seed = seed)
    #if (!is.null(report)) md.log("RF preimputation is done", date=debug, time=debug, trace=FALSE)
  }
  else if (tolower(preimpute) == "mm") {
    data <- meanmode(data)
    #if (!is.null(report)) md.log("Mean/Mode preimputation is done", date=debug, time=debug, trace=FALSE)
  }
  else stop(paste(preimpute, "is not recognized preimputation argument"))

  return(data)
}



