% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/refBasedCts.r
\name{refBasedCts}
\alias{refBasedCts}
\title{Reference based imputation of repeated measures continuous data}
\usage{
refBasedCts(
  obsData,
  outcomeVarStem,
  nVisits,
  trtVar,
  baselineVars = NULL,
  baselineVisitInt = TRUE,
  type = "MAR",
  M = 5
)
}
\arguments{
\item{obsData}{The data frame to be imputed.}

\item{outcomeVarStem}{String for stem of outcome variable name, e.g. y if y1, y2, y3 are the outcome columns}

\item{nVisits}{The integer number of visits (not including baseline)}

\item{trtVar}{The string variable name of the randomised treatment group variable. The reference arm is assumed
to correspond to \code{trtVar==0}.}

\item{baselineVars}{A string or vector of strings specfying the baseline variables. Often this will include
the baseline measurement of the outcome}

\item{baselineVisitInt}{TRUE/FALSE indicating whether to allow for interactions between each baseline variable
and visit. Default is TRUE.}

\item{type}{A string specifying imputation type to use. Valid options are "MAR", "J2R"}

\item{M}{Number of imputations to generate.}
}
\value{
A list of imputed datasets, or if \code{M=1}, just the imputed data frame.
}
\description{
Performs multiple imputation of a repeatedly measured continuous endpoint in a
randomised clinical trial
using reference based imputation as proposed by \doi{10.1080/10543406.2013.834911}{Carpenter et al (2013)}. This approach
can be used for imputation of missing data in randomised clinical trials.
}
\details{
Unlike most implementations of reference based imputation, this implementation
imputes conditional on the maximum likelihood estimates of the model parameters,
rather than a posterior draw. If one is interested in frequentist valid inferences,
this is ok provided the bootstrapping used, for example with using the
\href{https://cran.r-project.org/package=bootImpute}{bootImpute} package.

Intermediate missing values are imputed assuming MAR, based on the mixed model fit
to that patient's treatment arm. Monotone missing values are imputed using the specified
imputation type.

Baseline covariates must be numeric variables. If you have factor variables you must
code these into suitable dummy indicators and pass these to the function.
}
\examples{
#take a look at ctsTrialWide data
head(ctsTrialWide)

#impute the missing outcome values twice assuming MAR
imps <- refBasedCts(ctsTrialWide, outcomeVarStem="y", nVisits=3, trtVar="trt",
                    baselineVars=c("v", "y0"), type="MAR", M=2)

#now impute using jump to reference method
imps <- refBasedCts(ctsTrialWide, outcomeVarStem="y", nVisits=3, trtVar="trt",
                    baselineVars=c("v", "y0"), type="J2R", M=2)

#for frequentist valid inferences we use bootstrapping from the bootImpute package
\dontrun{
  #bootstrap 10 times using 2 imputations per bootstrap. Note that to do this
  #we specify nImp=2 to bootImpute by M=1 to the refBasedCts function.
  #Also, 10 bootstraps is far too small to get reliable inferences. To do this
  #for real you would want to use a lot more (e.g. at least nBoot=1000).
  library(bootImpute)
  bootImps <- bootImpute(ctsTrialWide, refBasedCts, nBoot=10, nImp=2,
                         outcomeVarStem="y", nVisits=3, trtVar="trt",
                         baselineVars=c("v", "y0"), type="J2R", M=1)

  #write a small wrapper function to perform an ANCOVA at the final time point
  ancova <- function(inputData) {
    coef(lm(y3~v+y0+trt, data=inputData))
  }
  ests <- bootImputeAnalyse(bootImps, ancova)
  ests
}
}
\references{
Carpenter JR, Roger JH, Kenward MG. Analysis of Longitudinal Trials with Protocol Deviation:
A Framework for Relevant, Accessible Assumptions, and Inference via Multiple Imputation. (2013) 23(6) 1352-1371

von Hippel PT & Bartlett JW (2019) Maximum likelihood multiple imputation: Faster imputations and
consistent standard errors without posterior draws \href{https://arxiv.org/abs/1210.0870v10}{arXiv:1210.0870v10}.
}
