% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/mvtboost.R
\name{mvtb}
\alias{mvtb}
\title{Fitting a Multivariate Tree Boosting Model}
\usage{
mvtb(X, Y, n.trees = 100, shrinkage = 0.01, interaction.depth = 1,
  trainfrac = 1, bag.frac = 1, cv.folds = 1, s = NULL, seednum = NULL,
  compress = FALSE, save.cv = FALSE, mc.cores = 1, samp.iter = FALSE,
  alpha = 0.5, cov.discrep = 1, weight.type = 1, ...)
}
\arguments{
\item{X}{vector, matrix, or data.frame of predictors. For best performance, continuous predictors should be scaled to have unit variance. Categorical variables should converted to factors.}

\item{Y}{vector, matrix, or data.frame for outcome variables. Missing values must be imputed. To easily compare influences across outcomes and for numerical stability, outcome variables should be scaled to have unit variance.}

\item{n.trees}{maximum number of trees to be included in the model. Trees are grown until a minimum number observations in each node is reached. This minimum can be modified using additional arguments (below).}

\item{shrinkage}{a constant multiplier for the predictions from each tree to ensure a slow learning rate. Default is .01. Small shrinkage values may require a large number of trees to provide adequate fit.}

\item{interaction.depth}{fixed depth of trees to be included in the model. A tree depth of 1 corresponds to fitting stumps (main effects only), higher tree depths capture higher order interactions.}

\item{trainfrac}{proportion of the sample used for training the multivariate additive model. The number of trees that minimize the multivariate MSE in the test set are given in out$best.trees. If both cv.folds and trainfrac are specified, the CV is done within the training set only.}

\item{bag.frac}{proportion of the training sample used to fit univariate trees for each response at each iteration. Default: 1}

\item{cv.folds}{number of cross validation folds. Default: 1. Runs k + 1 models, where the k models are run in parallel and the final model is run on the entire sample. If larger than 1, the number of trees that minimize the multivariate MSE averaged over k-folds is reported in object$best.trees.}

\item{s}{vector of indices denoting observations to be used for the training sample. If s is specified, trainfrac is ignored.}

\item{seednum}{integer to ensure that results are reproducible}

\item{compress}{T/F. Compress output results list using bzip2 (approx 10\% of original size). Default FALSE.}

\item{save.cv}{T/F. Save all k-fold cross-validation models.}

\item{mc.cores}{Number of cores for cross validation.}

\item{samp.iter}{Experimental.}

\item{alpha}{optional argument to select predictors that explain more variance or covariance in outcomes. Variance reductions are weighted by alpha, and covariance reductions are weighted by 1-alpha.}

\item{cov.discrep}{Experimental. Choose the type of covariance discrepancy.}

\item{weight.type}{Experimental.}

\item{...}{additional arguments passed to \code{gbm}. These include \code{distribution}, \code{weights}, \code{var.monotone}, \code{n.minobsinnode}, \code{keep.data}, \code{verbose}, \code{class.stratify.cv}.  Note that other \code{distribution} arguments have not been tested.}
}
\value{
Fitted model. This is a list containing the following elements:

\itemize{
  \item models - list of gbm models for each outcome. Functions from the gbm package (e.g. to compute relative influence, print trees, obtain predictions, etc) can be directly applied to each of these models
  \item covex - covariance explained in each pair of outcomes by each predictor. The covariance explained is only unambiguous if predictors are independent, otherwise it is an approximation. If the interaction.depth is larger than 1, the covariance explained is attributed to the predictor with the largest effect.
  \item maxiter - maximum number of iterations run (the number of trees fit)
  \item best.trees - list of the best number of trees given by minimizing the multivariate MSE error in the test set, by CV, or just the last tree fit. Many of the functions in the package default to using the minimum value of the three.
  \item rel.infl - n x q x n.trees matrix of relative influences
  \item w.rel.infl - n x q x n.trees matrix of weighted relative influences (see details)
  \item params - arguments to mvtb
  \item trainerr - multivariate training error at each tree.
  \item testerr  - multivariate test error at each tree (if trainfrac < 1)
  \item cverr    - multivariate cv error at each tree (if cv.folds > 1)
  \item bestxs - matrix of predictors selected at each tree
  \item resid - n x q matrix of residuals after fitting all trees
  \item ocv - if save.cv=TRUE, returns the CV models.
  \item wm.raw - raw decreases in covariance attributable to a given tree
  \item wm.rel - relative decreases in covariance attributable to a given tree
  \item s - indices of training sample
  \item n - number of observations
  \item xnames
  \item ynames
}
}
\description{
Builds on \code{gbm} (Ridgeway 2013; Friedman, 2001) to fit a univariate tree model for each outcome, selecting predictors at each iteration that explain covariance between the outcomes. The number of trees included in the model can be chosen by minimizing the multivariate mean squared error using cross validation or a test set.
}
\details{
This function computes separate gbm models for each outcome (contained in \code{$models}). From these models, the covariance explained by
each predictor is then computed based on the reduction in covariance between outcomes that results from fitting a single tree to each outcome, one outcome at a time.
The reduction in covariance between each pair of outcomes due to splitting on each predictor over all trees is the 'covariance explained' by each predictor, and is recorded in \code{$covex}.

The rows (pairs of outcomes) and the columns (predictors) of \code{$covex} can be clustered so that groups of predictors that explain similar pairs of covariances are closer together (see  \code{mvtb.cluster}).
A simple heatmap of this matrix can be obtained by the function \code{mvtb.heat}. The \code{covex} by each predictor is only unambiguous if the predictors are uncorrelated and interaction.depth = 1.
If predictors are not independent, the decomposition of covariance explained is only approximate (like the decomposition of R^2 by each predictor in a linear model).
If interaction.depth > 1, the following heuristic is used: the covariance explained by the tree is assigned to the predictor with the largest influence in each tree.

(Relative) influences can be retrieved using \code{mvtb.ri}, which are the usual reductions in SSE due to splitting on each predictor. 'Weighted' influences for each predictor are the usual reductions in SSE weighted by
the covariance explained in all pairs of outcomes by that predictor. This allows predictor selection to be informed by the covariance explained. Higher weight can be given to explaining variances or covariances by controlling the parameter
\code{alpha}:  weight = \code{alpha}(varex) + (1-\code{alpha})(covex). By default, \code{alpha} = .5, equally weighting variance and covariance explained.
Setting \code{alpha} = 0 corresponds to weighting covariance explained only,
letting \code{alpha} = 1 corresponds to weighting variance explained only.

The model is tuned jointly by selecting the number of trees that minimize multivariate mean squared error in a test set (by setting \code{trainfrac}) or averaged over k folds in k-fold cross-validation (by setting \code{cv.folds > 1}).
The best number of trees is available via \code{$best.trees}. Cross-validation can be parallelized by setting mc.cores > 1.
If both \code{cv.folds} and \code{trainfrac} is specified, cross-validation is carried out within the training set.
Cross-validation models are usually discarded but can be saved by setting \code{save.cv = TRUE}. CV models can be accessed from \code{$ocv} of the
output object.
Multivariate mean squared training, test, and cv error are available from \code{$trainerr, $testerr, $cverr} from the output object.
Observations can be specifically set for inclusion in the training set by passing a vector of integers indexing the rows to include to \code{s}.
If \code{s} is specified, \code{trainfrac} is ignored but cross-validation will be carried out for observations in \code{s}.

Since the output objects can be large, automatic compression is available by setting \code{compress=TRUE}.
All methods that use the \code{mvtb} object automatically uncompress this object if necessary.
The function \code{mvtb.uncomp} is available to manually decompress the object.

Note that trees are grown until a minimum number of observations in each node is reached.
If the number of training samples*bag.fraction is less the minimum number of observations, (which can occur with small data sets), this will throw an error.
Adjust the \code{n.minobsinnode}, \code{trainfrac}, or \code{bag.fraction}.

This is a beta version with details subject to change. Any contributions are welcome.
}
\examples{
set.seed(123)
n <- 1000
X <- matrix(rbinom(n*3,size=1,prob=.5),n,3)    # create 3 dichotomous predictors
X2 <- cbind(x1x2=X[,1]*X[,2],x2x3=X[,2]*X[,3]) # create 2 interaction terms
Xf <- cbind(X,X2)                              # full design matrix, used for data generation
E <- matrix(rnorm(n*4),nrow=n,ncol=4)          # independent errors
B <- matrix(0,nrow=5,ncol=4)
B[4,1] <- 1      # x1x2 interaction has a true effect on outcome 1
B[5,3:4] <- 1    # x2x3 interaction has a true effect on outcomes 3 and 4
Y <- Xf \%*\% B + E


out <- mvtb(
 X=X,                   # matrix of predictors
 Y=Y,                   # matrix of responses
 n.trees=100,           # number of trees
 shrinkage=.1,          # shrinkage or learning rate
 interaction.depth = 5, # number of splits in each tree
 bag.frac = .5          # bagging fraction
 )

summary(out)
plot(out)
mvtb.nonlin(out,X=X,Y=Y)
mvtb.perspec(out)
mvtb.cluster(out)
mvtb.heat(out)
}
\references{
Miller P.J., Lubke G.H, McArtor D.B., Bergeman C.S. (Submitted) Finding structure in data with multivariate tree boosting.

Ridgeway, G., Southworth, M. H., & RUnit, S. (2013). Package 'gbm'. Viitattu, 10, 2013.

Elith, J., Leathwick, J. R., & Hastie, T. (2008). A working guide to boosted regression trees. Journal of Animal Ecology, 77(4), 802-813.

Friedman, J. H. (2001). Greedy function approximation: a gradient boosting machine. Annals of statistics, 1189-1232.
}
\seealso{
\code{summary.mvtb}, \code{predict.mvtb}, \code{mvtb.nonlin} to help detect nonlinear effects, \code{plot.mvtb}, \code{mvtb.perspec} for plots, \code{mvtb.cluster}, \code{mvtb.heat}
\code{mvtb.uncomp} to uncompress a compressed output object
}

