% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/netleague.R
\name{netleague}
\alias{netleague}
\alias{print.netleague}
\title{Create league table with network meta-analysis results}
\usage{
netleague(
  x,
  y,
  common = x$common,
  random = x$random,
  seq = x$seq,
  ci = TRUE,
  backtransf = TRUE,
  direct = FALSE,
  digits = gs("digits"),
  big.mark = gs("big.mark"),
  text.NA = ".",
  bracket = gs("CIbracket"),
  separator = gs("CIseparator"),
  lower.blank = gs("CIlower.blank"),
  upper.blank = gs("CIupper.blank"),
  writexl = !missing(path),
  path = "leaguetable.xlsx",
  overwrite = FALSE,
  warn.deprecated = gs("warn.deprecated"),
  ...
)

\method{print}{netleague}(
  x,
  common = x$x$common,
  random = x$x$random,
  warn.deprecated = gs("warn.deprecated"),
  ...
)
}
\arguments{
\item{x}{An object of class \code{netmeta} or \code{netleague}
(mandatory).}

\item{y}{An object of class \code{netmeta} (optional).}

\item{common}{A logical indicating whether a league table should be
printed for the common effects network meta-analysis.}

\item{random}{A logical indicating whether a league table should be
printed for the random effects network meta-analysis.}

\item{seq}{A character or numerical vector specifying the sequence
of treatments in rows and columns of a league table.}

\item{ci}{A logical indicating whether confidence intervals should
be shown.}

\item{backtransf}{A logical indicating whether printed results
should be back transformed. If \code{backtransf = TRUE}, results
for \code{sm = "OR"} are printed as odds ratios rather than log
odds ratios, for example.}

\item{direct}{A logical indicating whether league table with
network estimates (default) or estimates from direct comparisons
should be generated if argument \code{y} is not missing.}

\item{digits}{Minimal number of significant digits, see
\code{print.default}.}

\item{big.mark}{A character used as thousands separator.}

\item{text.NA}{A character string to label missing values.}

\item{bracket}{A character with bracket symbol to print lower
confidence interval: "[", "(", "\{", "".}

\item{separator}{A character string with information on separator
between lower and upper confidence interval.}

\item{lower.blank}{A logical indicating whether blanks between left
bracket and lower confidence limit should be printed.}

\item{upper.blank}{A logical indicating whether blanks between
separator and upper confidence limit should be printed.}

\item{writexl}{A logical indicating whether an Excel file should be
created (R package \bold{writexl} must be available).}

\item{path}{A character string specifying the filename of the Excel
file.}

\item{overwrite}{A logical indicating whether an existing Excel
file should be overwritten.}

\item{warn.deprecated}{A logical indicating whether warnings should
be printed if deprecated arguments are used.}

\item{\dots}{Additional arguments (passed on to \code{write_xlsx}
to create Excel file).}
}
\value{
An object of class \code{netleague} with corresponding \code{print}
function if \code{writexl = FALSE}. The object is a list containing
the league tables in list elements 'common' and 'random'. An Excel
file is created if \code{writexl = TRUE}. In this case, \code{NULL}
is returned in R.
}
\description{
A league table is a square matrix showing all pairwise comparisons
in a network meta-analysis. Typically, both treatment estimates and
confidence intervals are shown.
}
\details{
A league table is a square matrix showing all pairwise comparisons
in a network meta-analysis (Hutton et al., 2015). Typically, both
treatment estimates and confidence intervals are shown.

If argument \code{y} is not provided, the league table contains the
network estimates from network meta-analysis object \code{x} in the
lower triangle and the direct treatment estimates from pairwise
comparisons in the upper triangle. Note, for the random-effects
model, the direct treatment estimates are based on the common
between-study variance \eqn{\tau^2} from the network meta-analysis,
i.e. the square of list element \code{x$tau}.

If argument \code{y} is provided, the league table contains
information on treatment comparisons from network meta-analysis
object \code{x} in the lower triangle and from network
meta-analysis object \code{y} in the upper triangle. This is, for
example, useful to print information on efficacy and safety in the
same league table.

By default, an R object with the league tables is
generated. Alternatively, an Excel file is created if argument
\code{writexl = TRUE}.

This implementation reports pairwise comparisons of the treatment
in the column versus the treatment in the row in the lower triangle
and row versus column in the upper triangle. This is a common
presentation for network meta-analyses which allows to easily
compare direction and magnitude of treatment effects. For example,
given treatments A, B, and C, the results reported in the first row
and second column as well as second row and first column are from
the pairwise comparison A versus B. Note, this presentation is
different from the printout of a network meta-analysis object which
reports opposite pairwise comparisons in the lower and upper
triangle, e.g., A versus B in the first row and second column and B
versus A in the second row and first column.

If the same network meta-analysis object is used for arguments
\code{x} and \code{y}, reciprocal treatment estimates will be shown
in the upper triangle (see examples), e.g., the comparison B versus
A.

R function \code{\link{netrank}} can be used to change the order of
rows and columns in the league table (see examples).
}
\examples{
# Network meta-analysis of count mortality statistics
#
data(Woods2010)

p0 <- pairwise(treatment, event = r, n = N,
  studlab = author, data = Woods2010, sm = "OR")
net0 <- netmeta(p0)

oldopts <- options(width = 100)

# League table for common and random effects model with
# - network estimates in lower triangle
# - direct estimates in upper triangle
#
netleague(net0, digits = 2, bracket = "(", separator = " - ")

# League table for common effects model
#
netleague(net0, random = FALSE, digits = 2)

# Change order of treatments according to treatment ranking (random
# effects model)
#
netleague(net0, common = FALSE, digits = 2, seq = netrank(net0))
#
print(netrank(net0), common = FALSE)

\dontrun{
# Create a CSV file with league table for random effects model
#
league0 <- netleague(net0, digits = 2, bracket = "(", separator = " to ")
#
write.table(league0$random, file = "league0-random.csv",
  row.names = FALSE, col.names = FALSE, sep = ",")
#
# Create Excel files with league tables
# (if R package writexl is available)
#
netleague(net0, digits = 2, bracket = "(", separator = " to ",
          path = tempfile(fileext = ".xlsx"))
}

\donttest{
# Use depression dataset
#
data(Linde2015)

# Define order of treatments
#
trts <- c("TCA", "SSRI", "SNRI", "NRI",
  "Low-dose SARI", "NaSSa", "rMAO-A", "Hypericum", "Placebo")

# Outcome labels
#
outcomes <- c("Early response", "Early remission")

# (1) Early response
#
p1 <- pairwise(treat = list(treatment1, treatment2, treatment3),
  event = list(resp1, resp2, resp3), n = list(n1, n2, n3),
  studlab = id, data = Linde2015, sm = "OR")
#
net1 <- netmeta(p1, common = FALSE,
                seq = trts, ref = "Placebo", small = "bad")

# (2) Early remission
#
p2 <- pairwise(treat = list(treatment1, treatment2, treatment3),
  event = list(remi1, remi2, remi3), n = list(n1, n2, n3),
  studlab = id, data = Linde2015, sm = "OR")
#
net2 <- netmeta(p2, common = FALSE,
                seq = trts, ref = "Placebo", small = "bad")

options(width = 200)
netleague(net1, digits = 2)

netleague(net1, digits = 2, ci = FALSE)
netleague(net2, digits = 2, ci = FALSE)

# League table for two outcomes with
# - network estimates of first outcome in lower triangle
# - network estimates of second outcome in upper triangle
#
netleague(net1, net2, digits = 2, ci = FALSE)

netleague(net1, net2, seq = netrank(net1), ci = FALSE)
netleague(net1, net2, seq = netrank(net2), ci = FALSE)

print(netrank(net1))
print(netrank(net2))


# Report results for network meta-analysis twice
#
netleague(net1, net1, seq = netrank(net1), ci = FALSE,
  backtransf = FALSE)
netleague(net1, net1, seq = netrank(net1), ci = FALSE,
  backtransf = FALSE, direct = TRUE)
}

options(oldopts)

\dontrun{
# Generate a partial order of treatment rankings 
#
np <- netposet(net1, net2, outcomes = outcomes)

# Requires R package 'hasse'
#
hasse(np)
plot(np)
}

}
\references{
Hutton B, Salanti G, Caldwell DM, et al. (2015):
The PRISMA Extension Statement for Reporting of Systematic Reviews
Incorporating Network Meta-analyses of Health Care Interventions:
Checklist and Explanations.
\emph{Annals of Internal Medicine},
\bold{162}, 777
}
\seealso{
\code{\link{netmeta}}, \code{\link{netposet}},
  \code{\link{netrank}}
}
\author{
Guido Schwarzer \email{sc@imbi.uni-freiburg.de}, Gerta
  Rücker \email{ruecker@imbi.uni-freiburg.de}
}
\keyword{print}
