% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MCMC_build.R
\name{buildMCMC}
\alias{buildMCMC}
\title{Create an MCMC function from a NIMBLE model, or an MCMC configuration object}
\usage{
buildMCMC(conf, ...)
}
\arguments{
\item{conf}{An MCMC configuration object of class \code{MCMCconf} that specifies the model, samplers, monitors, and thinning intervals for the resulting MCMC function.  See \code{configureMCMC} for details of creating MCMC configuration objects.  Alternatively, \code{conf} may a NIMBLE model object, in which case an MCMC function corresponding to the default MCMC configuration for this model is returned.}

\item{...}{Additional arguments to be passed to \code{configureMCMC} if \code{conf} is a NIMBLE model object}
}
\description{
First required argument, which may be of class \code{MCMCconf} (an MCMC configuration object), or inherit from class \code{modelBaseClass} (a NIMBLE model object).  Returns an uncompiled executable MCMC function.  See details.
}
\details{
Calling buildMCMC(conf) will produce an uncompiled MCMC function object.  The uncompiled MCMC function will have arguments:

\code{niter}: The number of iterations to run the MCMC.

\code{thin}: The thinning interval for the \code{monitors} that were specified in the MCMC configuration.  If this argument is provided at MCMC runtime, it will take precedence over the \code{thin} interval that was specified in the MCMC configuration.  If omitted, the \code{thin} interval from the MCMC configuration will be used.

\code{thin2}: The thinning interval for the second set of monitors (\code{monitors2}) that were specified in the MCMC configuration.  If this argument is provided at MCMC runtime, it will take precedence over the \code{thin2} interval that was specified in the MCMC configuration.  If omitted, the \code{thin2} interval from the MCMC configuration will be used.

\code{reset}: Boolean specifying whether to reset the internal MCMC sampling algorithms to their initial state (in terms of self-adapting tuning parameters), and begin recording posterior sample chains anew. Specifying \code{reset = FALSE} allows the MCMC algorithm to continue running from where it left off, appending additional posterior samples to the already existing sample chains. Generally, \code{reset = FALSE} should only be used when the MCMC has already been run (default = TRUE).

\code{simulateAll}: Boolean specifying whether to simulate into all stochastic nodes.  This will overwrite the current values in all stochastic nodes (default = FALSE).

\code{time}: Boolean specifying whether to record runtimes of the individual internal MCMC samplers.  When \code{time = TRUE}, a vector of runtimes (measured in seconds) can be extracted from the MCMC using the method \code{mcmc$getTimes()} (default = FALSE).

\code{progressBar}: Boolean specifying whether to display a progress bar during MCMC execution (default = TRUE).  The progress bar can be permanently disabled by setting the system option \code{nimbleOptions(MCMCprogressBar = FALSE)}.

Samples corresponding to the \code{monitors} and \code{monitors2} from the MCMCconf are stored into the interval variables \code{mvSamples} and \code{mvSamples2}, respectively.
These may be accessed and converted into R matrix objects via:
\code{as.matrix(mcmc$mvSamples)}
\code{as.matrix(mcmc$mvSamples2)}

The uncompiled MCMC function may be compiled to a compiled MCMC object, taking care to compile in the same project as the R model object, using:
\code{Cmcmc <- compileNimble(Rmcmc, project = Rmodel)}

The compiled function will function identically to the uncompiled object, except acting on the compiled model object.
}
\section{Calculating WAIC}{


After the MCMC has been run, calling the \code{calculateWAIC()} method of the MCMC object will return the WAIC for the model, calculated using the posterior samples from the MCMC run.

\code{calculateWAIC()} accepts a single arugment:

\code{nburnin}: The number of pre-thinning MCMC samples to remove from the beginning of the posterior samples for WAIC calculation (default = 0).

The \code{calculateWAIC} method can only be used if the \code{enableWAIC} 
argument to \code{configureMCMC} or to \code{buildMCMC} is set to \code{TRUE}, or if the NIMBLE option
\code{enableWAIC} is set to \code{TRUE}.  If a user attempts
to call \code{calculateWAIC} without having set \code{enableWAIC = TRUE}
(either in the call to \code{configureMCMC}, or \code{buildMCMC}, or as a NIMBLE option),
an error will occur.  

The \code{calculateWAIC} method calculates the WAIC of the model that the
MCMC was performed on. The WAIC (Watanabe, 2010) is calculated from
Equations 5, 12, and 13 in Gelman (2014) (i.e. using \emph{p}WAIC2).  The set
of all stochastic nodes monitored by the MCMC object will be treated as
\eqn{theta} for the purposes of e.g. Equation 5 from Gelman (2014). 
All non-monitored nodes downstream of the monitored nodes that are necessary
to calculate \eqn{p(y|theta)} will be simulated from the posterior samples of 
\eqn{theta}.  This allows customization of exactly what predictive 
distribution \eqn{p(y|theta)} to use for calculations.  For more detail
on the use of different predictive distributions, see Section 2.5 from Gelman
(2014).  

Note that there exist sets of monitored parameters that do not lead to valid
WAIC calculations.  Specifically, for a valid WAIC calculation, every 
node that a data node depends on must be either monitored, or be
downstream from monitored nodes.  An easy way to ensure this is satisfied
is to monitor all top-level parameters in a model (NIMBLE's default).  
Another way to guarantee correctness is to monitor all nodes
directly upstream from a data node. However, other combinations of monitored
nodes are also valid.  If \code{enableWAIC = TRUE}, NIMBLE checks to see if
the set of monitored nodes is valid, and returns an error if not.
}
\examples{
\dontrun{
code <- nimbleCode({
    mu ~ dnorm(0, 1)
    x ~ dnorm(mu, 1)
    y ~ dnorm(x, 1)
})
Rmodel <- nimbleModel(code, data = list(y = 0))
conf <- configureMCMC(Rmodel)
Rmcmc <- buildMCMC(conf, enableWAIC = TRUE)
Cmodel <- compileNimble(Rmodel)
Cmcmc <- compileNimble(Rmcmc, project=Rmodel)
Cmcmc$run(10000)
samples <- as.matrix(Cmcmc$mvSamples)
head(samples)
WAIC <- Cmcmc$calculateWAIC(nburnin = 1000)
}

}
\author{
Daniel Turek
}
\references{
Watanabe, S. (2010). Asymptotic equivalence of Bayes cross validation and widely applicable information criterion in singular learning theory. \emph{Journal of Machine Learning Research} 11: 3571-3594.

Gelman, A., Hwang, J. and Vehtari, A. (2014). Understanding predictive information criteria for Bayesian models. \emph{Statistics and Computing} 24(6): 997-1016.
}
\seealso{
\code{\link{configureMCMC}} \code{\link{runMCMC}} \code{\link{nimbleMCMC}}
}

