% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multinomial-ml.R
\name{multinomial_ml}
\alias{multinomial_ml}
\title{Multinomial Machine Learning}
\usage{
multinomial_ml(y = NULL, X = NULL, learner = "forest", scale = TRUE)
}
\arguments{
\item{y}{Outcome vector.}

\item{X}{Covariate matrix (no intercept).}

\item{learner}{String, either \code{"forest"} or \code{"l1"}. Selects the base learner to estimate each expectation.}

\item{scale}{Logical, whether to scale the covariates. Ignored if \code{learner} is not \code{"l1"}.}
}
\value{
Object of class \code{mml}.
}
\description{
Estimation strategy to estimate conditional choice probabilities for ordered non-numeric outcomes.
}
\details{
Multinomial machine learning expresses conditional choice probabilities as expectations of binary variables:

\deqn{p_m \left( X_i \right) = \mathbb{E} \left[ 1 \left( Y_i = m \right) | X_i \right]}

This allows us to estimate each expectation separately using any regression algorithm to get an estimate of conditional probabilities.\cr

\code{\link{multinomial_ml}} combines this strategy with either regression forests or penalized logistic regression with an L1 penalty,
according to the user-specified parameter \code{learner}.\cr

If \code{learner == "l1"}, the penalty parameters are chosen via 10-fold cross-validation 
and \code{\link[stats]{model.matrix}} is used to handle non-numeric covariates. Additionally, if \code{scale == TRUE}, the covariates are scaled to 
have zero mean and unit variance.
}
\examples{
\donttest{
## Load data from orf package.
set.seed(1986)

library(orf)
data(odata)
odata <- odata[1:100, ] # Subset to reduce elapsed time.

y <- as.numeric(odata[, 1])
X <- as.matrix(odata[, -1])

## Training-test split.
train_idx <- sample(seq_len(length(y)), floor(length(y) * 0.5))

y_tr <- y[train_idx]
X_tr <- X[train_idx, ]

y_test <- y[-train_idx]
X_test <- X[-train_idx, ]

## Fit multinomial machine learning on training sample using two different learners.
multinomial_forest <- multinomial_ml(y_tr, X_tr, learner = "forest")
multinomial_l1 <- multinomial_ml(y_tr, X_tr, learner = "l1")

## Predict out of sample.
predictions_forest <- predict(multinomial_forest, X_test)
predictions_l1 <- predict(multinomial_l1, X_test)

## Compare predictions.
cbind(head(predictions_forest), head(predictions_l1))}

}
\seealso{
\code{\link{ordered_ml}}, \code{\link{ocf}}
}
\author{
Riccardo Di Francesco
}
