% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/markers.R
\name{markers}
\alias{markers}
\alias{marker}
\alias{addMarker}
\alias{setMarkers}
\alias{modifyMarker}
\alias{getMarkers}
\alias{removeMarkers}
\alias{swapGenotypes}
\alias{modifyMarkerMatrix}
\title{Marker functions}
\usage{
marker(x, ..., allelematrix, alleles = NULL, afreq = NULL, missing = 0,
  chrom = NA, pos = NA, name = NA, mutmat = NULL)

addMarker(x, m, ...)

setMarkers(x, m, annotations = NULL, missing = 0)

modifyMarker(x, marker, ids, genotype, alleles, afreq, chrom, name, pos)

getMarkers(x, markernames = NULL, chroms = NULL, fromPos = NULL,
  toPos = NULL)

removeMarkers(x, markers = NULL, markernames = NULL, chroms = NULL,
  fromPos = NULL, toPos = NULL)

swapGenotypes(x, ids)

modifyMarkerMatrix(x, ids, new.alleles)
}
\arguments{
\item{x}{a \code{\link{linkdat}} object}

\item{...}{an even number of vectors, indicating individuals and their
genotypes. See examples.}

\item{allelematrix}{a matrix with one row per pedigree member and two
columns per marker, containing the alleles for a single marker.}

\item{alleles}{a numeric or character vector containing allele names.}

\item{afreq}{a numerical vector with allele frequencies. An error is given
if they don't sum to 1 (rounded to 3 decimals).}

\item{missing}{a numeric - or character - of length 1, indicating the code
for missing alleles.}

\item{chrom}{NA or an integer (the chromosome number of the marker).}

\item{pos}{NA or a non-negative real number indicating the genetic position
(in cM) of the marker.}

\item{name}{NA or a character (the name of the marker).}

\item{mutmat}{a mutation matrix, or a list of two such matrices named
'female' and 'male'. The matrix/matrices must be square, with the allele
labels as dimnames, and each row must sum to 1 (after rounding to 3
decimals).}

\item{m}{a \code{marker} object or a matrix with alleles. (In
\code{setMarkers} this matrix can contain data of several markers.)}

\item{annotations}{a list of marker annotations.}

\item{marker, markers}{a numeric indicating which marker(s) to use/modify.}

\item{ids}{a numeric indicating individual(s) to be modified. In
\code{swapGenotypes} this must have length 2.}

\item{genotype}{a vector of length 1 or 2, containing the genotype to be
given the \code{ids} individuals. See examples.}

\item{markernames}{NULL or a character vector.}

\item{chroms}{NULL or a numeric vector of chromosome numbers.}

\item{fromPos, toPos}{NULL or a single numeric.}

\item{new.alleles}{a numerical matrix of dimensions \code{length(ids),
2*x$nMark}. Entries refer to the internal allele numbering.}
}
\value{
The \code{marker} function returns an object of class \code{marker}:
This is a numerical 2-column matrix with one row per individual, and
attributes 'alleles' (a character vector with allele names), 'nalleles' (the
number of alleles) and 'missing' (the input symbol for missing marker
alleles), 'chrom' (chromosome number), 'name' (marker identifier), 'pos'
(chromosome position in cM).

For \code{addMarker}, \code{setMarkers}, \code{removeMarkers},
\code{modifyMarker}, \code{modifyMarkerMatrix} and \code{swapGenotypes}, a
\code{linkdat} object is returned, whose \code{markerdata} element has been
set/modified.

For \code{getMarkers} a numeric vector containing marker numbers (i.e. their
indices in \code{x$markerdata}) for the markers whose 'name' attribute is
contained in \code{markernames}, 'chrom' attribute is contained in
\code{chroms}, and 'pos' attribute is between \code{from} and \code{to}.
NULL arguments are skipped, so \code{getMarkers(x)} will return
\code{seq_len(x$nMark)} (i.e. all markers).
}
\description{
Functions for setting and manipulating marker genotypes for 'linkdat'
objects.
}
\examples{

x = linkdat(toyped)
x = removeMarkers(x, 1) # removing the only marker.
x

# Creating and adding a SNP marker with alleles 'a' and 'b', for which 
# individual 1 is heterozygous, individuals 2 and 4 are homozygous for the 
# 'b' allele, and individual 3 has a missing genotype.
m1 = marker(x, 1, c('a','b'), c(2,4), c('b','b')) 
x = addMarker(x, m1)

# A rare SNP for which both children are heterozygous.
# The 'alleles' argument can be skipped, but is recommended to ensure 
# correct order of the frequencies. 
m2 = marker(x, 3:4, 1:2, alleles=1:2, afreq=c(0.99, 0.01))
x = addMarker(x, m2)

# Modifying the second marker:
# Making it triallelic, and adding a genotype to the father.
x = modifyMarker(x, marker=2, alleles=1:3, ids=1, genotype=2:3)

# Adding an empty SNP (all genotypes are missing):
x = addMarker(x, 0, alleles=c('A', 'B'))

# Similar shortcut for creating a marker for which all 
# pedigree members are homozygous for an allele (say 'b'):
x = addMarker(x, 'b') 
# Alternative: m = marker(x, 'b'); addMarker(x, m)


}
\seealso{
\code{\link{linkdat}}
}
\author{
Magnus Dehli Vigeland
}
