% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitCF.R
\name{fitCF}
\alias{fitCF}
\title{Fit a model of Clutch Frequency for marine turtles.}
\usage{
fitCF(
  x = c(mu = 4, sd = 100, p = 0),
  fixed.parameters = NULL,
  data = stop("Data formated with TableECFOCF() must be provided"),
  method = c("Nelder-Mead", "BFGS"),
  control = list(trace = 1, REPORT = 100, maxit = 500),
  itnmax = c(500, 100),
  hessian = TRUE,
  parallel = TRUE,
  verbose = FALSE
)
}
\arguments{
\item{x}{Initial parameters to be fitted}

\item{fixed.parameters}{Parameters that are fixed.}

\item{data}{CMR data formated using TableECFOCF()}

\item{method}{Method to be used by optimx()}

\item{control}{List of controls for optimx()}

\item{itnmax}{A vector with maximum iterations for each method.}

\item{hessian}{Logical to estimate SE of parameters}

\item{parallel}{If TRUE, will use parallel computing for ECFOCF_f()}

\item{verbose}{If TRUE, print the parameters at each step}
}
\value{
Return a list of class ECFOCF with the fit information.\cr
The list has the following items:\cr
\itemize{
\item \code{data}: The observations to be fitted
\item \code{par}: The fitted parameters
\item \code{SE}: The standard error of parameters if hessian is TRUE
\item \code{value}: The -log likelihood of observations within the fitted model
\item \code{AIC}: The AIC of fitted model
\item \code{mu}: The vector of fitted mu values
\item \code{sd}: The vector of fitted sd values
\item \code{prob}: The vector of fitted capture probabilities
\item \code{a}: The vector of fitted capture probabilities multiplier
\item \code{OTN}: The vector of fitted relative probabilities of contribution
\item \code{period_categories}: A list with the different period probabilities as named vectors for each category
\item \code{period}: The combined period probabilities using OTN as named vector
\item \code{CF_categories}: A list with the different CF probabilities as named vectors for each category
\item \code{CF}: The combined CF probabilities  using OTN as named vector
\item \code{ECFOCF_categories}: A list with the different probability ECFOCF tables for each category
\item \code{ECFOCF}: The combined table of ECFOCF  using OTN probabilities tables
\item \code{ECFOCF_0}: The combined table of ECFOCF probabilities tables  using OTN without the OCF=0
\item \code{SE_df}: A data.frame with SE and 95\\% confidence intervals for meanx and vx (mean and variance of clutch frequency for x category), OTNx (proportion for x category), and probx (capture probability for x category)
}
}
\description{
This function fits a model of clutch frequency.\cr
This model is an enhanced version of the one published by Briane et al. (2007).\cr
Parameters are \code{mu} and \code{sd} being the parameters of a
distribution used to model the clutch frequency.\cr
This distribution is used only as a guide but has not statistical meaning.\cr
The parameter \code{p} is the -logit probability that a female is seen
on the beach for a particular nesting event. It includes both the probability
that it is captured but also the probability that it uses that specific beach.\cr
Several categories of females can be included in the model using index after
the name of the parameter, for example \code{mu1}, \code{sd1} and \code{mu2},
\code{sd2} indicates that two categories of females with different clutch
frequencies distribution are present. Similarly \code{p1} and \code{p2} indicates
that two categories of females with different capture probabilities are present.\cr
If more than one category is used, then it is necessary to include the
parameter \code{OTN} to indicate the relative frequencies of each category.
If two categories are used, one \code{OTN} parameter named \code{ONT1} must
be included. The \code{OTN2} is forced to be 1. Then the relative frequency
for category 1 is \code{OTN1/(OTN1+1)} and for category 2 is \code{1/(OTN1+1)}.
Same logic must be applied for 3 and more categories with always the last one
being fixed to 1.\cr

if p or a (logit of the capture probability) are equal to -Inf,
the probability of capture is 0 and if they are equal to
+Inf, the probability is 1.\cr

The value of p out of the period
of nesting must be set to +Inf (capture probability=1)
to indicate that no turtle is nesting in this period.\cr

p must be set to -Inf (capture probability=0) to indicate that no
monitoring has been done during a specific period of the nesting season.\cr

The best way to indicate capture probability for 3D model (OCF, ECF, Period)
is to indicate p.period common for all categories and a1, a2, etc for each category.
The capture probability for category 1 will be p.period * a1, and for category 2
will be p.period * a2, etc. \cr

In this case, the parameters p.period should be indicated in fitted parameters
as well as a1, but a2 must be fixed to +Inf in fixed.parameters. Then the capture
probability for category 2 will be p.period and for category 1 a1 * p.period.\cr

If itnmax is equal to 0, it will return the model using the parameters without fitting them.\cr
}
\details{
fitCF fit a model of Clutch Frequency for marine turtles.
}
\examples{
\dontrun{
library(phenology)
# Example
data(MarineTurtles_2002)
ECFOCF_2002 <- TableECFOCF(MarineTurtles_2002)

# Parametric model for clutch frequency
o_mu1p1_CFp <- fitCF(x = c(mu = 2.1653229641404539, 
                 sd = 1.1465246643327098, 
                 p = 0.25785366120357966), 
                 fixed.parameters=NULL, 
                 data=ECFOCF_2002, hessian = TRUE)
 
# Non parametric model for clutch frequency
o_mu1p1_CFnp <- fitCF(x = c(mu.1 = 18.246619595610383, 
                       mu.2 = 4.2702163522832892, 
                       mu.3 = 2.6289986859556458, 
                       mu.4 = 3.2496360919228611, 
                       mu.5 = 2.1602522716550943, 
                       mu.6 = 0.68617023351032846, 
                       mu.7 = 4.2623607001877026, 
                       mu.8 = 1.1805600042630455, 
                       mu.9 = 2.2786176350939731, 
                       mu.10 = 0.47676265496204945, 
                       mu.11 = 5.8988238539197062e-08, 
                       mu.12 = 1.4003187851424953e-07, 
                       mu.13 = 2.4128444894899776e-07, 
                       mu.14 = 2.4223748020049825e-07, 
                       p = 0.32094401970037578), 
                 fixed.parameters=c(mu.15 = 1E-10), 
                 data=ECFOCF_2002, hessian = TRUE)
                 
o_mu2p1 <- fitCF(x = c(mu1 = 1.2190766766978423, 
                     sd1 = 0.80646454821956925, 
                     mu2 = 7.1886819592223246, 
                     sd2 = 0.18152887523015518, 
                     p = 0.29347220802963259, 
                     OTN = 2.9137627675219533), 
                  fixed.parameters=NULL,
                  data=ECFOCF_2002, hessian = TRUE)

o_mu1p2 <- fitCF(x = c(mu = 5.3628701816871462, 
                     sd = 0.39390555498088764, 
                     p1 = 0.61159637544418755, 
                     p2 = -2.4212753004659189, 
                     OTN = 0.31898004668901009),
                 data=ECFOCF_2002, hessian = TRUE)
                 
o_mu2p2 <- fitCF(x = c(mu1 = 0.043692606004492131, 
                   sd1 = 1.9446036983033428, 
                   mu2 = 7.3007868915644751, 
                   sd2 = 0.16109296152913491, 
                   p1 = 1.6860260469536992, 
                   p2 = -0.096816113083788985, 
                   OTN = 2.2604431232973501), 
                  data=ECFOCF_2002, hessian = TRUE)

compare_AIC(mu1p1=o_mu1p1_CFp, 
            mu2p1=o_mu2p1, 
            mu1p2=o_mu1p2, 
            mu2p2=o_mu2p2)
                 
o_mu3p3 <- fitCF(x = c(mu1 = 0.24286312214288761, 
                            sd1 = 0.34542255091729313, 
                            mu2 = 5.0817174343025551, 
                            sd2 = 1.87435099405695, 
                            mu3 = 5.2009265101740683, 
                            sd3 = 1.79700447678357, 
                            p1 = 8.8961708614726156, 
                            p2 = 0.94790116453886453, 
                            p3 = -0.76572930634505421, 
                            OTN1 = 1.2936848663276974, 
                            OTN2 = 0.81164278235645926),
                 data=ECFOCF_2002, hessian = TRUE)
                 

o_mu3p1 <- fitCF(x = structure(c(0.24387978183477, 
                                   1.2639261745506, 
                                   4.94288464711349, 
                                   1.945082889758, 
                                   4.9431672350811, 
                                   1.287663104591, 
                                   0.323636536050397, 
                                   1.37072039291397, 
                                   9.28055412564559e-06), 
                                  .Names = c("mu1", "sd1", "mu2", 
                                             "sd2", "mu3", "sd3", 
                                             "p", "OTN1", "OTN2")),
                 data=ECFOCF_2002, hessian = TRUE)
                 

o_mu1p3 <- fitCF(x = structure(c(4.65792402108387, 
                                   1.58445909785, 
                                   -2.35414198317177, 
                                   0.623757854800649, 
                                   -3.62623634029326, 
                                   11.6950204755787, 
                                   4.05273728846523), 
                                   .Names = c("mu", "sd", 
                                              "p1", "p2", "p3", 
                                              "OTN1", "OTN2")),
                 data=ECFOCF_2002, hessian = TRUE)
                 
compare_AIC(mu1p1=o_mu1p1, 
            mu2p1=o_mu2p1, 
            mu1p2=o_mu1p2, 
            mu2p2=o_mu2p2, 
            mu3p3=o_mu3p3, 
            mu1p3=o_mu1p3, 
            mu3p1=o_mu3p1)
            
 # 3D model for (ECF, OCF, period)           
            
ECFOCF_2002 <- TableECFOCF(MarineTurtles_2002, 
                           date0=as.Date("2002-01-01"))

fp <- rep(0, dim(ECFOCF_2002)[3])
names(fp) <- paste0("p.", formatC(1:(dim(ECFOCF_2002)[3]), width=2, flag="0"))
par <- c(mu = 2.6404831115214353, 
        sd = 0.69362774786433479, 
        mu_season = 12.6404831115214353, 
        sd_season = 1.69362774786433479)
par <- c(par, fp[attributes(ECFOCF_2002)$table["begin"]:
                 attributes(ECFOCF_2002)$table["final"]])
                 
# The value of p (logit of the capture probability) out of the period 
# of nesting must be set to +Inf (capture probability=1)
# to indicate that no turtle is nesting in this period

# p must be set to -Inf (capture probability=0) to indicate that no
# monitoring has been done during a specific period of the nesting season.

fixed.parameters <- c(p=+Inf)
# The fitted values are:
par <- c(mu = 2.4911638591178051, 
         sd = 0.96855483039640977, 
         mu_season = 13.836059118657793, 
         sd_season = 0.17440085345943984, 
         p.10 = 1.3348233607728222, 
         p.11 = 1.1960387774393837, 
         p.12 = 0.63025680979544774, 
         p.13 = 0.38648155002707452, 
         p.14 = 0.31547864054366048, 
         p.15 = 0.19720001827017075, 
         p.16 = 0.083199496372073328, 
         p.17 = 0.32969130595897905, 
         p.18 = 0.36582777525265819, 
         p.19 = 0.30301248314170637, 
         p.20 = 0.69993987591518514, 
         p.21 = 0.13642423871641118, 
         p.22 = -1.3949268190534629)

o_mu1p1season1 <- fitCF(x=par, data=ECFOCF_2002, 
                        fixed.parameters=fixed.parameters)

# Same model but with two different models of capture probabilities
                        
fp <- rep(0, dim(ECFOCF_2002)[3])
names(fp) <- paste0("p1.", formatC(1:(dim(ECFOCF_2002)[3]), width=2, flag="0"))
par <- c(mu = 2.6404831115214353, 
        sd = 0.69362774786433479, 
        mu_season = 12.6404831115214353, 
        sd_season = 1.69362774786433479)
par <- c(par, fp[attributes(ECFOCF_2002)$table["begin"]:
                 attributes(ECFOCF_2002)$table["final"]])
names(fp) <- paste0("p2.", formatC(1:(dim(ECFOCF_2002)[3]), width=2, flag="0"))
par <- c(par, fp[attributes(ECFOCF_2002)$table["begin"]:
                 attributes(ECFOCF_2002)$table["final"]])
fixed.parameters <- c(p1=+Inf, p2=+Inf)

o_mu1p2season1 <- fitCF(x=par, data=ECFOCF_2002, 
                        fixed.parameters=fixed.parameters)

# Here the two different capture probabilities are different 
# by a constant:
# p1=invlogit(-p)     [Note that invlogit(-a1) = 1]
# p2=invlogit(-p)*invlogit(-a2)

fp <- rep(0, dim(ECFOCF_2002)[3])
names(fp) <- paste0("p.", formatC(1:(dim(ECFOCF_2002)[3]), width=2, flag="0"))
par <- c(mu = 2.6404831115214353, 
        sd = 0.69362774786433479, 
        mu_season = 12.6404831115214353, 
        sd_season = 1.69362774786433479,  
        a2=0)
par <- c(par, fp[attributes(ECFOCF_2002)$table["begin"]:
                 attributes(ECFOCF_2002)$table["final"]])
fixed.parameters <- c(a1=+Inf, p=+Inf)

o_mu1p1aseason1 <- fitCF(x=par, data=ECFOCF_2002, 
                        fixed.parameters=fixed.parameters)
                        data=ECFOCF_2002)
 
}
}
\seealso{
Briane J-P, Rivalan P, Girondot M (2007) The inverse problem applied
to the Observed Clutch Frequency of Leatherbacks from Yalimapo beach,
French Guiana. Chelonian Conservation and Biology 6:63-69

Fossette S, Kelle L, Girondot M, Goverse E, Hilterman ML, Verhage B,
Thoisy B, de, Georges J-Y (2008) The world's largest leatherback
rookeries: A review of conservation-oriented research in French
Guiana/Suriname and Gabon. Journal of Experimental Marine Biology
and Ecology 356:69-82

Other Model of Clutch Frequency: 
\code{\link{ECFOCF_f}()},
\code{\link{ECFOCF_full}()},
\code{\link{TableECFOCF}()},
\code{\link{fitCF_MHmcmc}()},
\code{\link{fitCF_MHmcmc_p}()},
\code{\link{generateCF}()},
\code{\link{lnLCF}()},
\code{\link{logLik.ECFOCF}()},
\code{\link{plot.ECFOCF}()},
\code{\link{plot.TableECFOCF}()}
}
\author{
Marc Girondot \email{marc.girondot@gmail.com}
}
\concept{Model of Clutch Frequency}
