% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/col_vals_expr.R
\name{col_vals_expr}
\alias{col_vals_expr}
\alias{expect_col_vals_expr}
\alias{test_col_vals_expr}
\title{Do column data agree with a predicate expression?}
\usage{
col_vals_expr(
  x,
  expr,
  preconditions = NULL,
  actions = NULL,
  step_id = NULL,
  brief = NULL,
  active = TRUE
)

expect_col_vals_expr(object, expr, preconditions = NULL, threshold = 1)

test_col_vals_expr(object, expr, preconditions = NULL, threshold = 1)
}
\arguments{
\item{x}{A data frame, tibble (\code{tbl_df} or \code{tbl_dbi}), Spark DataFrame
(\code{tbl_spark}), or, an agent object of class \code{ptblank_agent} that is created
with \code{\link[=create_agent]{create_agent()}}.}

\item{expr}{An expression to use for this test. This can either be in the
form of a call made with the \code{expr()} function or as a one-sided \strong{R}
formula (using a leading \code{~}).}

\item{preconditions}{expressions used for mutating the input table before
proceeding with the validation. This is ideally as a one-sided R formula
using a leading \code{~}. In the formula representation, the \code{.} serves as the
input data table to be transformed (e.g.,
\code{~ . \%>\% dplyr::mutate(col = col + 10)}.}

\item{actions}{A list containing threshold levels so that the validation step
can react accordingly when exceeding the set levels. This is to be created
with the \code{\link[=action_levels]{action_levels()}} helper function.}

\item{step_id}{One or more optional identifiers for the single or multiple
validation steps generated from calling a validation function. The use of
step IDs serves to distinguish validation steps from each other and provide
an opportunity for supplying a more meaningful label compared to the step
index. By default this is \code{NULL}, and \strong{pointblank} will automatically
generate the step ID value (based on the step index) in this case. One or
more values can be provided, and the exact number of ID values should (1)
match the number of validation steps that the validation function call will
produce (influenced by the number of \code{columns} provided), (2) be an ID
string not used in any previous validation step, and (3) be a vector with
unique values.}

\item{brief}{An optional, text-based description for the validation step.}

\item{active}{A logical value indicating whether the validation step should
be active. If the step function is working with an agent, \code{FALSE} will make
the validation step inactive (still reporting its presence and keeping
indexes for the steps unchanged). If the step function will be operating
directly on data, then any step with \code{active = FALSE} will simply pass the
data through with no validation whatsoever. The default for this is \code{TRUE}.}

\item{object}{A data frame, tibble (\code{tbl_df} or \code{tbl_dbi}), or Spark
DataFrame (\code{tbl_spark}) that serves as the target table for the expectation
function or the test function.}

\item{threshold}{A simple failure threshold value for use with the
expectation function. By default, this is set to \code{1} meaning that any
single unit of failure in data validation results in an overall test
failure. Whole numbers beyond \code{1} indicate that any failing units up to
that absolute threshold value will result in a succeeding \strong{testthat}
test. Likewise, fractional values (between \code{0} and \code{1}) act as a
proportional failure threshold.}
}
\value{
For the validation function, the return value is either a
\code{ptblank_agent} object or a table object (depending on whether an agent
object or a table was passed to \code{x}). The expectation function invisibly
returns its input but, in the context of testing data, the function is
called primarily for its potential side-effects (e.g., signaling failure).
The test function returns a logical value.
}
\description{
The \code{col_vals_expr()} validation function checks for whether column values in
a table match a user-defined predicate expression. The validation function
can be used directly on a data table or with an \emph{agent} object (technically,
a \code{ptblank_agent} object) whereas the expectation and test functions can only
be used with a data table. The types of data tables that can be used include
data frames, tibbles, database tables (\code{tbl_dbi}), and Spark DataFrames
(\code{tbl_spark}). Each validation step or expectation will operate over the
number of test units that is equal to the number of rows in the table (after
any \code{preconditions} have been applied).
}
\details{
Having table \code{preconditions} means \strong{pointblank} will mutate the table just
before interrogation. Such a table mutation is isolated in scope to the
validation step(s) produced by the validation function call. Using
\strong{dplyr} code is suggested here since the statements can be translated to
SQL if necessary. The code is most easily supplied as a one-sided \strong{R}
formula (using a leading \code{~}). In the formula representation, the \code{.} serves
as the input data table to be transformed (e.g.,
\code{~ . \%>\% dplyr::mutate(col_a = col_b + 10)}). Alternatively, a function could
instead be supplied (e.g.,
\code{function(x) dplyr::mutate(x, col_a = col_b + 10)}).

Often, we will want to specify \code{actions} for the validation. This argument,
present in every validation function, takes a specially-crafted list
object that is best produced by the \code{\link[=action_levels]{action_levels()}} function. Read that
function's documentation for the lowdown on how to create reactions to
above-threshold failure levels in validation. The basic gist is that you'll
want at least a single threshold level (specified as either the fraction of
test units failed, or, an absolute value), often using the \code{warn_at}
argument. This is especially true when \code{x} is a table object because,
otherwise, nothing happens. For the \verb{col_vals_*()}-type functions, using
\code{action_levels(warn_at = 0.25)} or \code{action_levels(stop_at = 0.25)} are good
choices depending on the situation (the first produces a warning when a
quarter of the total test units fails, the other \code{stop()}s at the same
threshold level).

Want to describe this validation step in some detail? Keep in mind that this
is only useful if \code{x} is an \emph{agent}. If that's the case, \code{brief} the agent
with some text that fits. Don't worry if you don't want to do it. The
\emph{autobrief} protocol is kicked in when \code{brief = NULL} and a simple brief will
then be automatically generated.
}
\section{Function ID}{

2-25
}

\examples{
# For all of the examples here, we'll
# use a simple table with three numeric
# columns (`a`, `b`, and `c`) and three
# character columns (`d`, `e`, and `f`)
tbl <-
  dplyr::tibble(
    a = c(1, 2, 1, 7, 8, 6),
    b = c(0, 0, 0, 1, 1, 1),
    c = c(0.5, 0.3, 0.8, 1.4, 1.9, 1.2),
  )
  
tbl

# A: Using an `agent` with validation
#    functions and then `interrogate()` 

# Validate that values in column `a`
# are integer-like by using the R modulo
# operator and expecting `0`
agent <-
  create_agent(tbl) \%>\%
  col_vals_expr(expr(a \%\% 1 == 0)) \%>\%
  interrogate()

# Determine if this validation
# had no failing test units (there
# are 6 test units, one for each row)
all_passed(agent)

# Calling `agent` in the console
# prints the agent's report; but we
# can get a `gt_tbl` object directly
# with `get_agent_report(agent)`

# B: Using the validation function
#    directly on the data (no `agent`)

# This way of using validation functions
# acts as a data filter: data is passed
# through but should `stop()` if there
# is a single test unit failing; the
# behavior of side effects can be
# customized with the `actions` option
tbl \%>\% 
  col_vals_expr(expr(a \%\% 1 == 0)) \%>\%
  dplyr::pull(a)
  
# C: Using the expectation function

# With the `expect_*()` form, we would
# typically perform one validation at a
# time; this is primarily used in
# testthat tests
expect_col_vals_expr(tbl, ~ a \%\% 1 == 0)

# D: Using the test function

# With the `test_*()` form, we should
# get a single logical value returned
# to us
test_col_vals_expr(tbl, ~ a \%\% 1 == 0)

# Variations

# We can do more complex things by
# taking advantage of the `case_when()`
# and `between()` functions (available
# for use in the pointblank package) 
tbl \%>\%
  test_col_vals_expr(~ case_when(
    b == 0 ~ a \%>\% between(0, 5) & c < 1,
    b == 1 ~ a > 5 & c >= 1
  ))

# If you only want to test a subset of
# rows, then the `case_when()` statement
# doesn't need to be exhaustive; any
# rows that don't fall into the cases
# will be pruned (giving us less test
# units overall) 
tbl \%>\%
  test_col_vals_expr(~ case_when(
    b == 1 ~ a > 5 & c >= 1
  ))

}
\seealso{
Other validation functions: 
\code{\link{col_exists}()},
\code{\link{col_is_character}()},
\code{\link{col_is_date}()},
\code{\link{col_is_factor}()},
\code{\link{col_is_integer}()},
\code{\link{col_is_logical}()},
\code{\link{col_is_numeric}()},
\code{\link{col_is_posix}()},
\code{\link{col_schema_match}()},
\code{\link{col_vals_between}()},
\code{\link{col_vals_equal}()},
\code{\link{col_vals_gte}()},
\code{\link{col_vals_gt}()},
\code{\link{col_vals_in_set}()},
\code{\link{col_vals_lte}()},
\code{\link{col_vals_lt}()},
\code{\link{col_vals_not_between}()},
\code{\link{col_vals_not_equal}()},
\code{\link{col_vals_not_in_set}()},
\code{\link{col_vals_not_null}()},
\code{\link{col_vals_null}()},
\code{\link{col_vals_regex}()},
\code{\link{conjointly}()},
\code{\link{rows_distinct}()}
}
\concept{validation functions}
