% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_policy_graph.R
\name{plot_policy_graph}
\alias{plot_policy_graph}
\alias{plot}
\title{Visualize a POMDP Policy Graph}
\usage{
plot_policy_graph(
  x,
  belief = TRUE,
  legend = TRUE,
  engine = c("igraph", "visNetwork"),
  col = NULL,
  ...
)
}
\arguments{
\item{x}{object of class POMDP containing a solved POMDP problem.}

\item{belief}{logical; display belief proportions as a pie chart in each node. This requires belief space sampling and may be slow.}

\item{legend}{logical; display a legend for colors used belief proportions?}

\item{engine}{The plotting engine to be used.}

\item{col}{colors used for the states.}

\item{\dots}{plotting options passed on to the plotting engine (see Details
section).}
}
\description{
The function plots the POMDP policy graph in a POMDP. It
uses \code{plot} in \pkg{igraph} with appropriate plotting options.
}
\details{
The function currently only plots converged policy graphs.

The policy graph nodes represent segments in the value function. Each
segment represents one or more believe states. The pie chart in each node
(if available) represent the average belief proportions of the belief states
belonging to the node/segment.

The built in plotting engines are \pkg{igraph} and \pkg{visNetwork}. The
additional arguments specified in \code{...} are passed on to the engine
plotting function.  For \pkg{igraph} this is
\code{\link[igraph]{plot.igraph}} (see \code{\link[igraph]{plot.common}} for
available options). For \pkg{visNetwork} this is
\code{\link[visNetwork]{visIgraph}}.

Other plotting libraries can be used by creating a policy graph (as an
igraph object) using \code{\link{policy_graph}} and converting it into a
suitable representation for that library.
}
\examples{

data("Tiger")
sol <- solve_POMDP(model = Tiger)
sol

## policy graph
policy_graph(sol)

## visualization
plot_policy_graph(sol)

## use a different graph layout (circle and manual; needs igraph)
library("igraph")
plot_policy_graph(sol, layout = layout.circle)
plot_policy_graph(sol, layout = rbind(c(1,1), c(1,-1), c(0,0), c(-1,-1), c(-1,1)))

## hide labels and legend
plot_policy_graph(sol, edge.label = NA, vertex.label = NA, legend = FALSE)

## add a plot title
plot_policy_graph(sol, main = sol$model$name)

## custom larger vertex labels (A, B, ...)
plot_policy_graph(sol, 
  vertex.label = LETTERS[1:nrow(policy(sol)[[1]])], 
  vertex.label.cex = 2,
  vertex.label.color = "white")

## plotting using the graph object 
## (e.g., using the graph in the layout and to change the edge curvature)
pg <- policy_graph(sol)
plot(pg, 
  layout = layout_as_tree(pg, root = 3, mode = "out"),
  edge.curved = curve_multiple(pg, .2))

## changes labels
plot(pg, 
  edge.label = abbreviate(E(pg)$label),
  vertex.label = sol$solution$pg$action,
  vertex.size = 10)

## plot interactive graphs using the visNetwork library
plot_policy_graph(sol, engine = "visNetwork")

## add smooth edges and a layout (note, engine can be abbreviated)
plot_policy_graph(sol, engine = "vis", layout = "layout_in_circle", smooth = TRUE)
}
\seealso{
\code{\link{solve_POMDP}}, \code{\link{policy_graph}}.

From: \pkg{igraph} \code{\link[igraph]{plot.igraph}},
\code{\link[igraph]{igraph_options}}, \code{\link[igraph]{plot.common}}.

From \pkg{visNetwork}: \code{\link[visNetwork]{visIgraph}}.
}
\keyword{hplot}
