% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weighted_table.R
\name{makeWeightsDT}
\alias{makeWeightsDT}
\title{Make a \code{data.table} of Tabulated, Aggregated Values and Weights}
\usage{
makeWeightsDT(data, values = NULL, print = NULL, adjust = NULL,
  formula = NULL, Surv.response = TRUE, by.other = NULL,
  custom.levels = NULL, custom.levels.cut.low = NULL, weights = NULL,
  internal.weights.values = NULL, enclos = NULL, NA.text = NULL)
}
\arguments{
\item{data}{DF/DT; passed to \code{envir} in \code{eval}}

\item{values}{values to tabulate. Anything \code{evalPopArg} can evaluate.}

\item{print}{variables to tabulate by and include in \code{prVars} in attributes}

\item{adjust}{variables to tabulate by and include in \code{adVars} in attributes}

\item{formula}{a formula such as \code{fot ~ sex} or \code{Surv(fot, lex.Xst) ~ sex}}

\item{Surv.response}{logical, if \code{TRUE} throws error if response in
\code{formula} is not a \code{Surv} object and vice versa}

\item{by.other}{other variables to tabulate by and include 
in \code{boVars} in attributes}

\item{custom.levels}{a named list of values. When "inflating" the data
in the cross-join / cartesian join sense (try e.g. \code{merge(1:5, 1:2)}),
one can supply the levels to inflate by using this to ensure inflation is full.
E.g. data might only have levels present to do inflation analogous to
\code{merge(2:5, 1:2)} although \code{merge(1:5, 1:2)} is intended and 
needed.}

\item{custom.levels.cut.low}{a character string vector of variable names.
These variables mentioned in \code{custom.levels} and existing in data
or first modified (in data) using \code{cutLow()} (essentially 
\code{cut()} with \code{right = FALSE} and returning the lower bounds
as values). Handy for aggregating data e.g. to survival intervals.
\strong{NOTE}: the appropriate elements in \code{custom.levels} for these 
variables must exceptionally contain an extra value as the roof used in
cutting, which will not be used in "inflating" the table using a merge.
See Examples.}

\item{weights}{a named list or long-form data.frame of weights. See Examples.}

\item{internal.weights.values}{the variable to use to compute internal
weights; only used if \code{weights = "internal"}.}

\item{enclos}{the enclosing environment passed on to \code{eval}. Variables
not found in \code{data} or searched for here.}

\item{NA.text}{a character string to display in a \code{warning}
if there are any rows with missing \code{values} or \code{adjust} values.
\strong{special:} key phrase \code{\%\%NA_COUNT\%\%} in text is replaced
with the count of missing observations.
E.g. \code{"Missing \%\%NA_COUNTS\%\% observations due to derpness."}}
}
\description{
An internal function that aggregates a table
and merges in weights.
}
\examples{
library(survival)
library(data.table)

makeWeightsDT <- popEpi:::makeWeightsDT ## this avoids errors during tests

sire <- copy(popEpi::sire)
set.seed(1L)
sire$sex <- rbinom(nrow(sire), 1, 0.5)
ag <- lexpand(sire, birth = "bi_date", entry = "dg_date", exit = "ex_date",
              status = status \%in\% 1:2, pophaz = popmort, pp = FALSE,
              aggre = list(sex, agegr = cut(dg_age, c(0,50,75,Inf)), fot), 
              fot = seq(0, 5, 1/12))
ps <- quote(list(sex, fot))
as <- quote(list(agegr))
vs <- list(quote(list(pyrs, at.risk)))
ws <- list(agegr = c(0.2,0.4,0.4))

#### custom.levels usage
fb <- seq(0, 5-1/12, 1/12) ## exclude 5 as no row has that value
ag2 <- ag[fot > 0.5,]
# repeats fot intervals < 0.5 as empty rows
# may be the safest way to do this
dt <- makeWeightsDT(ag2, print = ps, adjust = as, 
                    values = vs, weights = ws,
                    custom.levels = list(fot = fb))
## aggregate from intervals seq(0, 5, 1/12) to 0:5
fb2 <- 0:5 ## (this time we include 5 as the roof)       
dt <- makeWeightsDT(ag2, print = ps, adjust = as, 
                    values = vs, weights = ws,
                    custom.levels = list(fot = fb2),
                    custom.levels.cut.low = "fot")              
                    

#### use of enclos
TF <- environment()
gender <- factor(ag$sex)
dt <- makeWeightsDT(ag, print = quote(gender), adjust = as, 
                    values = vs, weights = ws, enclos = TF)
## or NULL: uses calling frame by default.
dt <- makeWeightsDT(ag, print = quote(gender), adjust = as, 
                    values = vs, weights = ws,
                    enclos = NULL)
## passing parent.fram(1) is the same thing (as below),
## but won't pass in testing these examples somehow (but work in real life)
# dt <- makeWeightsDT(ag, print = quote(gender), adjust = as, 
#                     values = vs, weights = ws,
#                     enclos = NULL)                  

#### formula usage
form <- Surv(fot, factor(from0to1))~gender
dt <- makeWeightsDT(ag, formula = form, Surv.response = TRUE,
                    adjust = as, values = vs, weights = ws,
                    enclos = NULL)
                    
## or
form <- Surv(fot, factor(from0to1))~gender + adjust(agegr)
dt <- makeWeightsDT(ag, formula = form, Surv.response = TRUE,
                    adjust = NULL, values = vs, weights = ws,
                    enclos = NULL)
                    
## or   
form <- from0to1 ~ fot + gender + adjust(agegr)
dt <- makeWeightsDT(ag, formula = form, Surv.response = FALSE,
                    adjust = NULL, values = vs, weights = ws,
                    enclos = NULL)            

form <- from0to1 ~ fot + adjust(agegr) + adjust(sex)
ws2 <- list(agegr = c(0.33, 0.33, 0.33), sex = c(0.5, 0.5))
dt <- makeWeightsDT(ag, formula = form, Surv.response = FALSE,
                    adjust = NULL, values = vs, weights = ws2,
                    enclos = NULL)

## international standard pops
ag <- lexpand(sire, birth = "bi_date", entry = "dg_date", exit = "ex_date",
              status = status \%in\% 1:2, pophaz = popmort, pp = FALSE,
              aggre = list(sex, agegr = cut(dg_age, c(seq(0, 85, 5), Inf)), fot), 
              fot = seq(0, 5, 1/12))
              
form <- from0to1 ~ fot + adjust(agegr)
dt <- makeWeightsDT(ag, formula = form, Surv.response = FALSE,
                    adjust = NULL, values = vs, weights = "world_1966_18of5",
                    enclos = NULL)
                    
form <- from0to1 ~ fot + adjust(agegr, sex)
dt <- makeWeightsDT(ag, formula = form, Surv.response = FALSE,
                    adjust = NULL, values = vs, 
                    weights = list(agegr = "nordic_2000_18of5", sex=c(1,1)),
                    enclos = NULL)
}
