% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fractional_years.R
\name{get.yrs}
\alias{get.yrs}
\title{Convert date objects to fractional years}
\usage{
get.yrs(x, year.length = "approx", ...)
}
\arguments{
\item{x}{a \code{Date} object, or anything that \code{link{as.Date}}
accepts}

\item{year.length}{character string, either \code{'actual'} or 
\code{'approx'}; can be abbreviated; see Details}

\item{...}{additional arguments passed on to \code{\link{as.Date}};
typically \code{format} when \code{x} is a character string variable,
and \code{origin} when \code{x} is numeric}
}
\description{
Using Date objects, calculates given 
dates as fractional years.
}
\details{
\code{x} should preferably be a \code{date}, \code{Date} or \code{IDate} 
object, although it can also be a character string variable 
which is coerced internally to \code{Date} format 
using \code{\link{as.Date.character}}.

When \code{ year.length = 'actual' }, fractional years are calculated as 
\code{ year + (day_in_year-1)/365 } for non-leap-years
and as \code{ year + (day_in_year-1)/366 } for leap years. 
If \code{ year.length = 'approx' }, fractional years are always
calculated as in \code{ year + (day_in_year-1)/365.242199 }. 

There is a slight difference, then, between the two methods
when calculating durations between fractional years. For
meticulous accuracy one might instead want to calculate durations using
dates (days) and convert the results to fractional years.
 
Note that dates are effectively converted to fractional years at 
\code{ 00:00:01 } o'clock:


\code{ get.yrs("2000-01-01") = 2000 }, and
\code{ get.yrs("2000-01-02") = 2000 + 1/365.242199 }.
}
\examples{

data("sire")
sire$dg_yrs <- get.yrs(sire$dg_date)
summary(sire$dg_yrs)

## see: ?as.Date.yrs
dg_date2 <- as.Date(sire$dg_yrs)
summary(as.numeric(dg_date2 - as.Date(sire$dg_date)))

## Epi's cal.yr versus get.yrs
d <- as.Date("2000-01-01")
Epi::cal.yr(d) ## 1999.999
get.yrs(d) ## 2000

## "..." passed on to as.Date, so character / numeric also accepted as input
## (and whatever else as.Date accepts)
get.yrs("2000-06-01")
get.yrs("20000601", format = "\%Y\%m\%d")
get.yrs("1/6/00", format = "\%d/\%m/\%y")

get.yrs(100, origin = "1970-01-01")


}
\seealso{
\code{\link[Epi]{cal.yr}}, \code{\link{as.Date.yrs}}, \code{\link{as.Date}}
}
\author{
Joonas Miettinen
}
