#' Retrieve a data frame of AUC scores
#'
#' The \code{auc} function takes an \code{S3} object generated by
#'   \code{\link{evalmod}} and retrieves a data frame with the Area Under
#'   the Curve (AUC) scores of ROC and Precision-Recall curves.
#'
#' @param curves An \code{S3} object generated by \code{\link{evalmod}}.
#'   The \code{auc} function accepts the following S3 objects.
#'
#'   \tabular{lll}{
#'     \strong{\code{S3} object}
#'     \tab \strong{# of models}
#'     \tab \strong{# of test datasets} \cr
#'
#'     sscurves \tab single   \tab single   \cr
#'     mscurves \tab multiple \tab single   \cr
#'     smcurves \tab single   \tab multiple \cr
#'     mmcurves \tab multiple \tab multiple
#'   }
#'
#'    See the \strong{Value} section of \code{\link{evalmod}} for more details.
#'
#' @return The \code{auc} function returns a data frame with AUC scores.
#'
#' @seealso \code{\link{evalmod}} for generating \code{S3} objects with
#'   performance evaluation measures. \code{\link{pauc}} for retieving a dataset
#'   of pAUCs.
#'
#' @examples
#'
#' #############################################################################
#' ### Single model & single test dataset
#' ###
#'
#' ## Load a dataset with 10 positives and 10 negatives
#' data(P10N10)
#'
#' ## Generate an sscurve object that contains ROC and Precision-Recall curves
#' sscurves <- evalmod(scores = P10N10$scores, labels = P10N10$labels)
#'
#' ## Shows AUCs
#' auc(sscurves)
#'
#'
#' #############################################################################
#' ### Multiple models & single test dataset
#' ###
#'
#' ## Create sample datasets with 100 positives and 100 negatives
#' samps <- create_sim_samples(1, 100, 100, "all")
#' mdat <- mmdata(samps[["scores"]], samps[["labels"]],
#'                modnames = samps[["modnames"]])
#'
#' ## Generate an mscurve object that contains ROC and Precision-Recall curves
#' mscurves <- evalmod(mdat)
#'
#' ## Shows AUCs
#' auc(mscurves)
#'
#'
#' #############################################################################
#' ### Single model & multiple test datasets
#' ###
#'
#' ## Create sample datasets with 100 positives and 100 negatives
#' samps <- create_sim_samples(10, 100, 100, "good_er")
#' mdat <- mmdata(samps[["scores"]], samps[["labels"]],
#'                modnames = samps[["modnames"]],
#'                dsids = samps[["dsids"]])
#'
#' ## Generate an smcurve object that contains ROC and Precision-Recall curves
#' smcurves <- evalmod(mdat, raw_curves = TRUE)
#'
#' ## Get AUCs
#' sm_aucs <- auc(smcurves)
#'
#' ## Get AUCs of Precision-Recall
#' sm_aucs_prc <- subset(sm_aucs, curvetypes == "PRC")
#'
#'
#' #############################################################################
#' ### Multiple models & multiple test datasets
#' ###
#'
#' ## Create sample datasets with 100 positives and 100 negatives
#' samps <- create_sim_samples(10, 100, 100, "all")
#' mdat <- mmdata(samps[["scores"]], samps[["labels"]],
#'                modnames = samps[["modnames"]],
#'                dsids = samps[["dsids"]])
#'
#' ## Generate an mscurve object that contains ROC and Precision-Recall curves
#' mmcurves <- evalmod(mdat, raw_curves = TRUE)
#'
#' ## Shows AUCs
#' mm_aucs <- auc(mmcurves)
#'
#' ## Get AUCs of Precision-Recall
#' mm_aucs_prc <- subset(mm_aucs, curvetypes == "PRC")
#'
#' @export
auc <- function(curves) UseMethod("auc")

#' @export
auc.default <- function(curves) {
  stop("An object of unknown class is specified")
}

#
# Print AUC scores
#
#' @rdname auc
#' @export
auc.aucs <- function(curves) {
  # Validation
  .validate(curves)

  # Return AUC scores
  attr(curves, "aucs")
}
